#include "util.h"

#include <maps/wikimap/mapspro/libs/social/feedback/workflow_logic.h>
#include <maps/wikimap/mapspro/libs/social/helpers.h>
#include <maps/wikimap/mapspro/libs/social/magic_strings.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/social/feedback/consts.h>
#include <yandex/maps/wiki/social/feedback/enums.h>
#include <yandex/maps/wiki/social/feedback/history_patch.h>
#include <yandex/maps/wiki/social/feedback/task.h>

namespace maps::wiki::social::feedback {

const std::string FIELDS_TO_SELECT = common::join(
    std::vector<std::string>{
        sql::col::ID,
        sql::col::CREATED_AT,
        sql::func::ST_ASBINARY + "(" + sql::col::POSITION + ") AS "
            + sql::col::POSITION,
        sql::col::TYPE,
        sql::col::BUCKET,
        sql::col::SOURCE,
        sql::col::DESCRIPTION,
        sql::col::ATTRS,
        sql::col::ACQUIRED_BY,
        sql::col::ACQUIRED_AT,
        sql::col::RESOLVED_BY,
        sql::col::REJECT_REASON,
        sql::col::RESOLVED_AT,
        sql::col::RESOLUTION,
        sql::col::HIDDEN,
        sql::col::INTERNAL_CONTENT,
        sql::col::DEPLOYED_AT,
        sql::col::OBJECT_ID,
        sql::col::HASH_VALUE,
        sql::col::DUPLICATE_HEAD_ID,
        sql::col::VIEWED_BY,
        sql::col::STATE_MODIFIED_AT,
        sql::col::PROCESSING_LVL,
        sql::col::INDOOR_LEVEL,
    },
    ","
);

void execUpdateTasks(
    pqxx::transaction_base& socialTxn,
    const std::string& setClause,
    const std::string& whereClause)
{
    std::stringstream query;

    query << "UPDATE " << sql::table::FEEDBACK_TASK
          << " SET " << setClause
          << " WHERE " << whereClause;
    // Can not use RETURNING, because of ON UPDATE TRIGGERS

    socialTxn.exec(query.str());
}

namespace {

std::string oldTimePoint()
{
    return "now() - " + OLD_FEEDBACK_TASK_SQL_INTERVAL;
}

std::string
ageTypeSqlCondition(
    AgeType ageType,
    const std::string& createdAtColumnOld,
    const std::string& createdAtColumnNew)
{
    switch (ageType) {
        case AgeType::Old:
            return
                putInBrackets(
                    whereClause(Workflow::Feedback) +
                    " AND " + createdAtColumnOld + " <= " + oldTimePoint()
                );
        case AgeType::New:
            return
                putInBrackets(
                    putInBrackets(
                        whereClause(Workflow::Feedback) +
                        " AND " + createdAtColumnNew + " > " + oldTimePoint()
                    ) +
                    " OR " +
                    whereClause(Workflow::Task)
                );
    }
}

} // namespace anonymous

std::string putInBrackets(const std::string& str)
{
    return "(" + str + ")";
}

std::string ageTypeSqlCondition(AgeType ageType)
{
    return ageTypeSqlCondition(
        ageType, sql::col::CREATED_AT, sql::col::CREATED_AT
    );
}

std::string ageTypeMatViewSqlCondition(AgeType ageType)
{
    return ageTypeSqlCondition(
        ageType, sql::col::MIN_CREATED_AT, sql::col::MAX_CREATED_AT
    );
}

namespace {

const std::string AGGREGATED_COUNT = "aggregated_count";

} // namespace

AggregatedCounter rowToAggregatedCounter(const pqxx::row& row)
{
    AggregatedCounter counter;
    for (const auto& field: row) {
        if (field.name() == AGGREGATED_COUNT) {
            counter.count = field.as<uint64_t>();
            continue;
        } else {
            switch (enum_io::fromString<Column>(field.name())) {
            case Column::Workflow:
                counter.workflow =
                    enum_io::fromString<Workflow>(field.as<std::string>());
                break;
            case Column::Source:
                counter.source = field.as<std::string>();
                break;
            case Column::Type:
                counter.type =
                    enum_io::fromString<Type>(field.as<std::string>());
                break;
            }
        }
    }
    return counter;
}

std::string getAggregatedCountQuery(
    const std::string& whereClause,
    const Columns& aggregationColumns)
{
    ASSERT(!aggregationColumns.empty());
    const std::string columns = common::join(aggregationColumns, ',');

    std::stringstream query;
    query << "WITH aggregation AS (SELECT "
        << feedback::workflowFieldClause() << " AS workflow"
        << ", source, type, COUNT(*) AS counter"
        << " FROM " << sql::table::FEEDBACK_TASK
        << " WHERE " << whereClause
        << " GROUP BY workflow, source, type)";

    query << "SELECT " << columns << ',';

    query << " SUM(counter) AS " << AGGREGATED_COUNT
          << " FROM aggregation";

    query << " GROUP BY " << columns;
    return query.str();
}

TasksBriefResult
constructTasksBriefResult(TasksBrief&& tasks, std::optional<size_t> limit)
{
    auto hasMore = HasMore::No;
    if (limit && tasks.size() > *limit) {
        hasMore = HasMore::Yes;
        tasks.pop_back();
    }

    return TasksBriefResult{std::move(tasks), hasMore};
}

} // namespace maps::wiki::social::feedback
