#include <yandex/maps/wiki/social/involvement.h>
#include <yandex/maps/wiki/social/involvement_stat.h>
#include <yandex/maps/wiki/social/involvement_filter.h>
#include <yandex/maps/wiki/social/common.h>
#include <yandex/maps/wiki/social/exception.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <maps/libs/chrono/include/time_point.h>
#include "helpers.h"
#include "magic_strings.h"

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/polygon.h>

#include <maps/libs/sql_chemistry/include/column_name.h>
#include <maps/libs/sql_chemistry/include/filter.h>

#include <cstddef>
#include <string>
#include <vector>

namespace maps::wiki::social {

namespace {

const std::string INVOLVEMENT_COLUMNS = common::join(
    std::vector<std::string>{
        sql::col::ID,
        sql::col::TITLE,
        sql::col::URL,
        sql::col::TYPES,
        sql::col::ENABLED,
        sql::col::START,
        sql::col::FINISH,
        sql::func::ST_ASBINARY + "(" + sql::col::POLYGONS + ") AS "
            + sql::col::POLYGONS
    },
    ","
);

const std::string INVOLVEMENT_COLUMNS_WITHOUT_PKEY = common::join(
    std::vector<std::string>{
        sql::col::TITLE,
        sql::col::URL,
        sql::col::TYPES,
        sql::col::ENABLED,
        sql::col::START,
        sql::col::FINISH,
        sql::col::POLYGONS
    },
    ","
);

Involvement loadFromPqxx(const pqxx::row& row)
{
    Involvement inv(
        row[sql::col::ID].as<TId>(),
        row[sql::col::TITLE].as<std::string>(),
        row[sql::col::URL].as<std::string>(),
        parseStringSetFromSqlArray(row[sql::col::TYPES].as<std::string>()),
        row[sql::col::ENABLED].as<bool>() ? Enabled::Yes : Enabled::No,
        chrono::parseSqlDateTime(row[sql::col::START].as<std::string>())
    );

    if (!row[sql::col::FINISH].is_null()) {
        inv.setFinish(
            chrono::parseSqlDateTime(row[sql::col::FINISH].as<std::string>())
        );
    }

    inv.setPolygons(
        geolib3::WKB::read<geolib3::MultiPolygon2>(
            pqxx::binarystring(row[sql::col::POLYGONS]).str()
        )
    );

    return inv;
}

} //anonymous namespace

Involvement::Involvement(
    TId id,
    std::string title,
    std::string url,
    std::set<std::string> types,
    Enabled enabled,
    chrono::TimePoint start
)
    : id_(id)
    , title_(std::move(title))
    , url_(std::move(url))
    , types_(std::move(types))
    , enabled_(enabled)
    , start_(start)
{
}

TId
Involvement::id() const
{
    return id_;
}

const std::string&
Involvement::title() const
{
    return title_;
}

const std::string&
Involvement::url() const
{
    return url_;
}

bool
Involvement::enabled() const
{
    return (enabled_ == Enabled::Yes);
}

const std::set<std::string>&
Involvement::types() const
{
    return types_;
}

chrono::TimePoint
Involvement::start() const
{
    return start_;
}

const std::optional<chrono::TimePoint>&
Involvement::finish() const
{
    return finish_;
}

const geolib3::MultiPolygon2&
Involvement::polygons() const
{
    return polygonsMerc_;
}

Involvement&
Involvement::setTitle(std::string title)
{
    title_ = std::move(title);
    return *this;
}

Involvement&
Involvement::setUrl(std::string url)
{
    url_ = std::move(url);
    return *this;
}

Involvement&
Involvement::setEnabled(Enabled enabled)
{
    enabled_ = enabled;
    return *this;
}

Involvement&
Involvement::setTypes(std::set<std::string> types)
{
    types_ = std::move(types);
    return *this;
}

Involvement&
Involvement::setStart(chrono::TimePoint start)
{
    start_ = start;
    return *this;
}

Involvement&
Involvement::setFinish(const std::optional<chrono::TimePoint>& finish)
{
    finish_ = finish;
    return *this;
}

Involvement&
Involvement::setPolygons(geolib3::MultiPolygon2 polygonsMerc)
{
    polygonsMerc_ = std::move(polygonsMerc);
    return *this;
}

Involvement
Involvement::byId(
    pqxx::transaction_base& socialTxn,
    TId id
)
{
    std::ostringstream query;
    query <<
        "SELECT " << INVOLVEMENT_COLUMNS << " " <<
        "FROM " << sql::table::INVOLVEMENTS << " " <<
        "WHERE " << sql::col::ID << " = " << id
    ;
    auto pqxxResult = socialTxn.exec(query.str());
    if (pqxxResult.empty()) {
        throw InvolvementNotFound() << "Involvement with id=" << id << " wasn't found";
    }
    ASSERT(pqxxResult.size() == 1);
    const auto& row = pqxxResult.front();
    return loadFromPqxx(row);
}

Involvements
Involvement::byFilter(
    pqxx::transaction_base& socialTxn,
    const InvolvementFilter& filter
)
{
    std::ostringstream query;
    query <<
        "SELECT " << INVOLVEMENT_COLUMNS << " " <<
        "FROM " << sql::table::INVOLVEMENTS << " " <<
        "WHERE " << sql::value::TRUE << " "
    ;
    if (filter.enabled_) {
        query <<
            " AND " <<
            (
                (filter.enabled_.value() == Enabled::Yes) ?
                    sql::col::ENABLED :
                    ("NOT " + sql::col::ENABLED)
            )
        ;
    }

    if (filter.startedBefore_) {
        std::string sqlTime = chrono::formatSqlDateTime(filter.startedBefore_.value());
        query <<
            " AND " <<
            sql::col::START << " < " << socialTxn.quote(sqlTime)
        ;
    }

    if (filter.startedAfter_) {
        std::string sqlTime = chrono::formatSqlDateTime(filter.startedAfter_.value());
        query <<
            " AND " <<
            sql::col::START << " > " << socialTxn.quote(sqlTime)
        ;
    }

    if (filter.finishedBefore_) {
        std::string sqlTime = chrono::formatSqlDateTime(filter.finishedBefore_.value());
        query <<
            " AND (" <<
                "(" << sql::col::FINISH << " IS NOT NULL " << ")" << " AND " <<
                "(" << sql::col::FINISH << " < " << socialTxn.quote(sqlTime) << ")" <<
            ")"
        ;
    }

    if (filter.finishedAfter_) {
        std::string sqlTime = chrono::formatSqlDateTime(filter.finishedAfter_.value());
        query <<
            " AND (" <<
                "(" << sql::col::FINISH << " IS NULL" << ")" << " OR " <<
                "(" << sql::col::FINISH << " > " << socialTxn.quote(sqlTime) << ")" <<
            ")"
        ;
    }

    if (filter.active_) {
        bool isActive = (filter.active_.value() == Active::Yes);
        query <<
            (isActive ? " AND " : " AND NOT " ) << "(" <<
                sql::col::ENABLED << " AND " <<
                sql::col::START << " < " << "now()" << " AND " <<
                "(" <<
                    "(" << sql::col::FINISH << " IS NULL" << ")" << " OR " <<
                    "(" << sql::col::FINISH << " > " << "now()" << ")" <<
                ")" <<
            ")"
        ;
    }

    /*
     * Involvement's empty multipolygon means that this involvements
     * works for the whole world. So all such involvements should present
     * in filtered result regardless of filtering bbox
    */
    if (filter.bboxMerc_) {
        const auto epsg3395 = geolib3::SpatialReference::Epsg3395;

        auto bboxFilter =
            sql_chemistry::GeometryFilter<epsg3395, geolib3::Polygon2>(
                sql_chemistry::ColumnName(sql::col::POLYGONS, {}),
                sql_chemistry::op::Geometry::Intersects,
                filter.bboxMerc_->polygon()
            )
            ||
            sql_chemistry::GeometryFilter<epsg3395, geolib3::MultiPolygon2>(
                sql_chemistry::ColumnName(sql::col::POLYGONS, {}),
                sql_chemistry::op::Geometry::Equals,
                geolib3::MultiPolygon2()
            )
        ;

        query << " AND (" << bboxFilter.serializeToSql(socialTxn) << ")";
    }

    auto pqxxResult = socialTxn.exec(query.str());
    Involvements result;
    for (const auto& row: pqxxResult) {
        result.emplace_back(loadFromPqxx(row));
    }
    return result;
}

void
Involvement::writeToDatabase(pqxx::transaction_base& socialTxn)
{
    std::ostringstream query;
    bool isNew = (id_ == 0);
    if (isNew) {
        query << "INSERT INTO " << sql::table::INVOLVEMENTS << " ";
    } else {
        query << "UPDATE " << sql::table::INVOLVEMENTS << " SET ";
    }
    query << "(" << INVOLVEMENT_COLUMNS_WITHOUT_PKEY << ")";

    if (isNew) {
        query << " VALUES ";
    } else {
        query << " = ";
    }

    query << "(" <<
        socialTxn.quote(title_) << ", " <<
        socialTxn.quote(url_) << ", " <<
        socialTxn.quote(dumpToSqlArray(types_)) << ", " <<
        ((enabled_ == Enabled::Yes) ? "TRUE" : "FALSE") << ", " <<
        socialTxn.quote(chrono::formatSqlDateTime(start_)) << ", " <<
        (
            finish_ ?
                socialTxn.quote(chrono::formatSqlDateTime(finish_.value())) :
                "NULL"
        ) << ", " <<
        makePqxxGeomExpr(socialTxn, polygonsMerc_) <<
    ") ";

    if (!isNew) {
        query << "WHERE " << sql::col::ID << " = " << id_ << " ";
    }
    query << "RETURNING " << sql::col::ID;

    auto pqxxResult = socialTxn.exec(query.str());
    if (
        (pqxxResult.size() != 1) ||
        (pqxxResult.front().size() != 1)
    ) {
        throw InvolvementNotFound() << "Involvement with id=" << id_ << " wasn't found";
    }
    id_ = pqxxResult.front().front().as<TId>();
}

InvolvementStatMap loadInvolvementStatMap(
    pqxx::transaction_base& socialTxn,
    const Involvements& involvements
)
{
    auto statsFromDb = InvolvementStat::byInvolvements(socialTxn, involvements);
    InvolvementStatMap result;
    for (const auto& involvement: involvements) {
        auto types = involvement.types();

        auto insertResult = result.insert({involvement, {}});
        if (!insertResult.second) {
            //repeated involvement found
            continue;
        }
        auto& stats = insertResult.first->second;

        for (auto& stat: statsFromDb) {
            if (
                (involvement.id() == stat.involvementId()) &&
                (types.count(stat.type()) > 0)
            ) {
                types.erase(stat.type());
                stats.emplace_back(stat);
            }
        }
        for (const auto& type: types) {
            stats.emplace_back(involvement.id(), type);
        }
    }
    return result;
}

} //namespace maps::wiki::social
