#include <yandex/maps/wiki/social/profile_gateway.h>

#include "helpers.h"
#include "magic_strings.h"

#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <boost/range/adaptor/map.hpp>

namespace maps::wiki::social {

namespace {

using ColumnToValue = std::map<std::string, std::string>;

ColumnToValue
generateColumnToValue(
    const ProfileOptionalFields& optionalFields,
    pqxx::transaction_base& txn)
{
    ColumnToValue res;

    if (optionalFields.about) {
        res[sql::col::ABOUT] = txn.quote(*optionalFields.about);
    }
    if (optionalFields.email) {
        res[sql::col::EMAIL] = txn.quote(*optionalFields.email);
    }
    if (optionalFields.hasBroadcastSubscription) {
        res[sql::col::BROADCAST_SUBSCRIPTION] = toPgValue(*optionalFields.hasBroadcastSubscription);
    }
    if (optionalFields.hasNewsSubscription) {
        res[sql::col::NEWS_SUBSCRIPTION] = toPgValue(*optionalFields.hasNewsSubscription);
        res[sql::col::NEWS_SUBSCRIPTION_MODIFIED_AT] = sql::value::NOW;
    }
    if (optionalFields.newsSubscriptionWelcomeMailSent) {
        res[sql::col::NEWS_SUBSCRIPTION_WELCOME_MAIL_SENT] = toPgValue(*optionalFields.newsSubscriptionWelcomeMailSent);
    }
    if (optionalFields.locale) {
        res[sql::col::LOCALE] = txn.quote(*optionalFields.locale);
    }

    return res;
}

} // unnamed namespace

ProfileGateway::ProfileGateway(pqxx::transaction_base& txn) :
    txn_(txn)
{}

std::optional<Profile> ProfileGateway::getUserProfile(TUid uid) const
{
    auto profiles = getUserProfiles({uid});
    return profiles.empty()
        ? std::nullopt
        : std::optional<Profile>{std::move(profiles.front())};
}

Profiles
ProfileGateway::getUserProfiles(const TUids& uids) const
{
    if (uids.empty()) {
        return {};
    }

    std::ostringstream query;
    query << " SELECT * FROM " << sql::table::PROFILE
          << " WHERE " + sql::col::UID + " IN ("
          << common::join(uids, ',')
          << ")";
    auto result = txn_.exec(query.str());
    Profiles profiles;
    for (const auto& row : result) {
        profiles.push_back(Profile(row));
    }
    return profiles;
}

Profiles ProfileGateway::getAllUserProfiles() const
{
    std::ostringstream query;
    query << " SELECT * "
             " FROM " << sql::table::PROFILE;
    auto result = txn_.exec(query.str());

    Profiles profiles;
    profiles.reserve(result.size());
    for (const auto& row : result) {
        profiles.emplace_back(Profile(row));
    }

    return profiles;
}

std::optional<Profile>
ProfileGateway::updateProfile(
    const TUid& uid,
    const ProfileOptionalFields& optionalFields)
{
    REQUIRE(!optionalFields.empty(), "No fields to update");

    auto columnToValue = generateColumnToValue(optionalFields, txn_);

    std::vector<std::string> setExprs;
    std::transform(
        columnToValue.begin(),
        columnToValue.end(),
        std::back_inserter(setExprs),
        [](const auto& singleColumnToValue) {
            return singleColumnToValue.first + " = " + singleColumnToValue.second;
        }
    );

    std::ostringstream query;
    query << " UPDATE " << sql::table::PROFILE << " SET " << common::join(setExprs, ',')
          << " WHERE " << sql::col::UID << " = " << uid
          << " RETURNING *";

    auto result = txn_.exec(query.str());
    ASSERT(result.size() <= 1);
    if (result.empty()) {
        return std::nullopt;
    }
    return Profile(result[0]);
}

Profile
ProfileGateway::insertProfile(
    const TUid& uid,
    const ProfileOptionalFields& optionalFields)
{
    auto columnToValue = generateColumnToValue(optionalFields, txn_);
    columnToValue[sql::col::UID] = std::to_string(uid);

    auto columns = boost::adaptors::keys(columnToValue);
    auto values = boost::adaptors::values(columnToValue);

    std::ostringstream query;
    query << "INSERT INTO " << sql::table::PROFILE << " (" << common::join(columns, ", ") << ") "
          << "VALUES (" << common::join(values, ',') << ") RETURNING *";

    try {
        auto result = txn_.exec(query.str());
        ASSERT(result.size() == 1);
        return Profile(result[0]);
    } catch (const pqxx::unique_violation& ex) {
        throw RuntimeError(ex.what());
    }
}

Profile
ProfileGateway::upsertProfile(
    const TUid& uid,
    const ProfileOptionalFields& optionalFields)
{
    auto updatedProfile = updateProfile(uid, optionalFields);
    return updatedProfile ? *updatedProfile : insertProfile(uid, optionalFields);
}

} // namespace maps::wiki::social
