#include <yandex/maps/wiki/social/super_moderation_console.h>

#include "helpers.h"
#include "tasks/close.h"
#include "tasks/count.h"
#include "tasks/defer.h"
#include "tasks/release.h"
#include "tasks/resolve.h"
#include "tasks/resolve_and_close.h"
#include "tasks/stats.h"

#include <yandex/maps/wiki/common/robot.h>


namespace maps::wiki::social {

SuperModerationConsole::SuperModerationConsole(
    pqxx::transaction_base& work,
    TUid uid
)
    : work_(work)
    , uid_(uid)
{
    checkUid(uid);
}

CountsByAoiCategoryId
SuperModerationConsole::countsByAoiCategoryId(
    ModerationMode mode,
    const EventFilter& eventFilter,
    const TIds& aoiIds,
    const ModerationTimeIntervals& moderationTimeIntervals) const
{
    checkAoi(aoiIds);
    return tasks::countsByAoiCategoryId(
        work_, uid_, mode, eventFilter,
        aoiIds, moderationTimeIntervals);
}

bool
SuperModerationConsole::hasAcquirableTasks(
    ModerationMode mode,
    const TIds& aoiIds,
    const ModerationTimeIntervals& moderationTimeIntervals) const
{
    checkAoi(aoiIds);
    return tasks::existsAcquirable(work_, uid_, mode, aoiIds, moderationTimeIntervals);
}

TaskIds
SuperModerationConsole::resolveTasksByTaskIds(
    ResolveResolution resolution, const TaskIds& taskIds) const
{
    return tasks::resolveByTaskIds(work_, uid_, resolution, taskIds);
}

TaskIds
SuperModerationConsole::resolveEditTasksByCommitIds(
    ResolveResolution resolution, const TIds& commitIds) const
{
    return tasks::resolveEditsByCommitIds(work_, uid_, resolution, commitIds);
}

TaskIds
SuperModerationConsole::closeTasksByTaskIds(
    CloseResolution resolution, const TaskIds& taskIds) const
{
    return tasks::closeResolvedTasksByTaskIds(work_, uid_, resolution, taskIds);
}

TaskIds
SuperModerationConsole::closeTasksByTaskIds(
    const TIds& taskIds,
    ResolveResolution resolveResolution,
    CloseResolution closeResolution) const
{
    return tasks::closeByTaskIds(
        work_, uid_, taskIds,
        resolveResolution, closeResolution);
}

TaskIds
SuperModerationConsole::closeEditTasksByCommitIds(
    const TIds& commitIds,
    ResolveResolution resolveResolution,
    CloseResolution closeResolution) const
{
    return tasks::closeEditsByCommitIds(
        work_, uid_, commitIds,
        resolveResolution, closeResolution);
}

TaskIds
SuperModerationConsole::closeRevertedTasks(
    const Tasks& tasks,
    CloseResolution closeResolution) const
{
    TaskIds taskIdsToClose;

    for (const auto& task: tasks) {
        if (task.isResolved() &&
            task.resolved().resolution() == ResolveResolution::Revert)
        {
            taskIdsToClose.emplace(task.id());
        }
    }

    if (!taskIdsToClose.empty()) {
        closeTasksByTaskIds(closeResolution, taskIdsToClose);
    }

    return taskIdsToClose;
}

TaskIds SuperModerationConsole::resolveTasksByUserAndCloseByRobot(
    const TaskIds& taskIds,
    ResolveResolution resolveResolution,
    CloseResolution closeResolution) const
{
    return tasks::resolveAndClose(
        work_, taskIds, uid_, uid_, resolveResolution, common::ROBOT_UID, closeResolution
    );
}

TaskIds SuperModerationConsole::resolveTasksByRobotAndCloseByUser(
    const TaskIds& taskIds,
    ResolveResolution resolveResolution,
    CloseResolution closeResolution) const
{
    return tasks::resolveAndClose(
        work_, taskIds, uid_, common::ROBOT_UID, resolveResolution, uid_, closeResolution
    );
}

TaskIds
SuperModerationConsole::deferTasksByTaskIds(
    const TaskIds& taskIds,
    const std::string& expiresAt) const
{
    checkExpires(expiresAt);

    auto releasedTaskIds = tasks::release(work_, uid_, taskIds);
    tasks::createDeferred(work_, uid_, releasedTaskIds, expiresAt);
    return releasedTaskIds;
}

size_t
SuperModerationConsole::todayProcessedCount(int tzInMinutes) const
{
    return tasks::todayProcessedTaskCountByUid(work_, uid_, tzInMinutes);
}

} // namespace maps::wiki::social
