#include "close.h"
#include "helpers.h"
#include <maps/wikimap/mapspro/libs/social/magic_strings.h>

#include <list>

namespace maps::wiki::social::tasks {

namespace {

const size_t BULK_SIZE = 1000;

const boost::format SELECT_RESOLVED_TASKS_BY_TASK_IDS(
    "SELECT " + sql::col::EVENT_ID +
    " FROM " + sql::table::TASK_ACTIVE +
    " WHERE " + sql::col::RESOLVE_RESOLUTION + " IS NOT NULL"
      " AND " + sql::col::LOCKED_BY + "=%1%"
      " AND " + sql::col::EVENT_ID + " IN (%2%)"
    " ORDER BY " + sql::col::EVENT_ID + " DESC FOR UPDATE");


const std::string CLOSE_TASKS_PREFIX =
    "UPDATE " + sql::table::TASK_ACTIVE + " SET ";

const std::string CLOSE_TASKS_POSTFIX =
        sql::col::CLOSE_RESOLUTION + "='%1%'," +
        sql::col::CLOSED_BY + "=%2%," +
        sql::col::CLOSED_AT + "=NOW()," +
        sql::col::LOCKED_BY + "=0," +
        sql::col::LOCKED_AT + "=NULL"
    " WHERE " + sql::col::EVENT_ID + " IN (%3%)";


const boost::format CLOSE_NOT_RESOLVED_TASKS(
    CLOSE_TASKS_PREFIX +
        sql::col::RESOLVE_RESOLUTION + "='%4%'," +
        sql::col::RESOLVED_BY + "=%2%," +
        sql::col::RESOLVED_AT + "=" + sql::value::NOW + "," +
    CLOSE_TASKS_POSTFIX);

const boost::format CLOSE_RESOLVED_TASKS(
    CLOSE_TASKS_PREFIX +
    CLOSE_TASKS_POSTFIX);


TaskIds
closeTasks(
    pqxx::transaction_base& work,
    TUid uid,
    const LockedTaskIds& data,
    ResolveResolution resolveResolution,
    CloseResolution closeResolution)
{
    if (!data.notResolved.empty()) {
        boost::format query(CLOSE_NOT_RESOLVED_TASKS);
        query % closeResolution % uid % common::join(data.notResolved, ',');
        query % resolveResolution;
        work.exec(query.str());
    }
    if (!data.resolved.empty()) {
        boost::format query(CLOSE_RESOLVED_TASKS);
        query % closeResolution % uid % common::join(data.resolved, ',');
        work.exec(query.str());
    }

    TaskIds processedTaskIds = data.notResolved;
    processedTaskIds.insert(data.resolved.begin(), data.resolved.end());
    return processedTaskIds;
}

} // namespace

TaskIds
closeResolvedTasksByTaskIds(
    pqxx::transaction_base& work,
    TUid uid,
    CloseResolution resolution,
    const TaskIds& taskIds)
{
    return selectAndUpdateTasks(
        work,
        uid,
        resolution,
        taskIds,
        SELECT_RESOLVED_TASKS_BY_TASK_IDS,
        CLOSE_RESOLVED_TASKS);
}

TaskIds
closeByTaskIds(
    pqxx::transaction_base& work,
    TUid uid,
    TIds taskIds,
    ResolveResolution resolveResolution,
    CloseResolution closeResolution)
{
    if (taskIds.size() <= BULK_SIZE) {
        auto lockedTaskIds = selectForUpdateTasksByTaskIds(work, taskIds);
        return closeTasks(
            work, uid, lockedTaskIds, resolveResolution, closeResolution);
    }

    std::list<TaskIds> reverseOrderBatches;
    while (!taskIds.empty()) {
        auto it = taskIds.begin();
        std::advance(it, std::min(taskIds.size(), BULK_SIZE));
        reverseOrderBatches.emplace_front(taskIds.begin(), it);
        taskIds.erase(taskIds.begin(), it);
    }

    for (const auto& batchTaskIds : reverseOrderBatches) {
        auto lockedTaskIds = selectForUpdateTasksByTaskIds(work, batchTaskIds);
        auto closedTaskIds = closeTasks(
            work, uid, lockedTaskIds, resolveResolution, closeResolution);
        taskIds.insert(closedTaskIds.begin(), closedTaskIds.end());
    }
    return taskIds;
}

TaskIds
closeEditsByCommitIds(
    pqxx::transaction_base& work,
    TUid uid,
    const TIds& commitIds,
    ResolveResolution resolveResolution,
    CloseResolution closeResolution)
{
    if (commitIds.size() <= BULK_SIZE) {
        auto lockedTaskIds = selectForUpdateEditTasksByCommitIds(work, commitIds);
        return closeTasks(
            work, uid, lockedTaskIds, resolveResolution, closeResolution);
    }

    return closeByTaskIds(
        work, uid, commitIdsToEditTaskIds(work, commitIds, BULK_SIZE),
        resolveResolution, closeResolution);
}

} // namespace maps::wiki::social::tasks
