#include "count.h"

#include <maps/wikimap/mapspro/libs/social/magic_strings.h>
#include "helpers.h"

#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/common/pg_utils.h>
#include <yandex/maps/wiki/social/event_filter.h>

namespace maps::wiki::social::tasks {

namespace {

const std::string COUNT = "COUNT";
const std::string COUNT1 = COUNT + "(1)";
const std::string IS_OWN = "is_own";
const std::string IS_OTHERS = "is_others";
const std::string IS_OLD = "is_old";

const std::string TASK_TABLE_ALIAS = "t.";
const std::string COMMIT_EVENT_TABLE_ALIAS = "ce.";

const std::set<std::string> COMMIT_EVENT_TYPES_STR {
    std::string(toString(EventType::Edit)),
    std::string(toString(EventType::Complaint)),
    std::string(toString(EventType::RequestForDeletion)),
};

std::string filterAoisClause(const TIds& aoiIds)
{
    ASSERT(!aoiIds.empty());
    return sql::col::AOI_ID + " " + common::sqlInCondition(aoiIds);
}

std::string
buildSelectFromTaskQuery(const std::vector<std::string>& fields)
{
    return
        " SELECT " +
            common::join(fields, ",") +
        " FROM " +
            sql::table::TASK_ACTIVE + " t" +
        " JOIN " +
            sql::table::AOI_FEED_TRUNK_TASK_ACTIVE + " a" +
            " USING (" + sql::col::EVENT_ID + " )";
}

std::map<TId, CountsByCategoryId>
convertRowsToAoiTaskCounts(const pqxx::result& rows)
{
    std::map<TId, CountsByCategoryId> aoiTaskCounts;

    for (const auto& row : rows) {
        auto count = row[COUNT].as<size_t>();
        auto& aoiCounts = aoiTaskCounts[row[sql::col::AOI_ID].as<TId>()];
        auto& categoryCounts =
            aoiCounts[row[sql::col::PRIMARY_OBJECT_CATEGORY_ID].as<std::string>({})];

        bool isOwn = row[IS_OWN].as<bool>();
        bool isOld = row[IS_OLD].as<bool>();
        bool isOthers = row[IS_OTHERS].as<bool>();

        size_t available = (!isOwn && !isOthers) ? count : 0;
        size_t acquired = isOwn ? count : 0;
        size_t old = isOld ? count : 0;
        size_t total = (isOthers ? count : 0) + acquired + available;

        categoryCounts += {available, acquired, old, total};
    }

    return aoiTaskCounts;
}

} // namespace

CountsByAoiCategoryId countsByAoiCategoryId(
    pqxx::transaction_base& work,
    TUid uid,
    ModerationMode mode,
    const EventFilter& eventFilter,
    const TIds& aoiIds,
    const ModerationTimeIntervals& moderationTimeIntervals)
{
    static const auto CAT = TASK_TABLE_ALIAS + sql::col::PRIMARY_OBJECT_CATEGORY_ID;
    static const auto AOI = sql::col::AOI_ID;

    const auto eventTypeFilterTransformed = eventFilter.eventType()
        ? filterParamClause(
              eventFilter.eventType(),
              TASK_TABLE_ALIAS, sql::col::TYPE, work
          )
        : filterParamClause(
              std::make_optional(COMMIT_EVENT_TYPES_STR),
              TASK_TABLE_ALIAS, sql::col::TYPE, work
          );

    const auto eventWhereFilter =
        commitEventTaskClause(mode, TASK_TABLE_ALIAS, moderationTimeIntervals) +
        eventTypeFilterTransformed +
        filterParamClause(
            eventFilter.createdBy(),
            COMMIT_EVENT_TABLE_ALIAS, sql::col::CREATED_BY, work
        ) +
        filterParamClause(
            eventFilter.objectId(),
            COMMIT_EVENT_TABLE_ALIAS, sql::col::PRIMARY_OBJECT_ID, work
        ) +
        primaryObjectCategoryIdClause(
            eventFilter,
            TASK_TABLE_ALIAS, work
        ) +
        filterParamClause(
            eventFilter.commitIds(),
            TASK_TABLE_ALIAS, sql::col::COMMIT_ID, work
        );

    const auto query =
        buildSelectFromTaskQuery({
            COUNT1,
            sql::value::TRUE + lockedClause(uid) + " AS " + IS_OWN,
            sql::value::TRUE + lockedByOthersClause(uid) + " AS " + IS_OTHERS,
            sql::value::TRUE + oldClause(mode, TASK_TABLE_ALIAS, moderationTimeIntervals) + " AS " + IS_OLD,
            CAT,
            AOI}) +
        (
            eventFilter.createdBy() || eventFilter.objectId()
                ? " JOIN " + sql::table::COMMIT_EVENT + " ce USING (" + sql::col::EVENT_ID + " )"
                : ""
        ) +
        " WHERE " +
            filterAoisClause(aoiIds) +
            deferredClause(eventFilter.deferred(), uid, TASK_TABLE_ALIAS) +
            eventWhereFilter +
        " GROUP BY " +
            common::join(
                std::vector<std::string>({IS_OWN, IS_OTHERS, IS_OLD, CAT, AOI}),
                ","
            );

    CountsByAoiCategoryId counts;
    counts.taskCounts = convertRowsToAoiTaskCounts(work.exec(query));
    counts.oldTaskAgeInHours = oldTaskAgeInHours(mode, moderationTimeIntervals);
    return counts;
}

bool existsAcquirable(
    pqxx::transaction_base& work,
    TUid uid,
    ModerationMode mode,
    const TIds& aoiIds,
    const ModerationTimeIntervals& moderationTimeIntervals)
{
    const auto query =
        buildSelectFromTaskQuery({sql::col::EVENT_ID}) +
        " WHERE " +
            filterAoisClause(aoiIds) +
            deferredClause(Deferred::No, uid, TASK_TABLE_ALIAS) +
            acquirableClause(uid) +
            commitEventTaskClause(mode, TASK_TABLE_ALIAS, moderationTimeIntervals) +
            filterParamClause(
                std::make_optional(COMMIT_EVENT_TYPES_STR),
                TASK_TABLE_ALIAS,
                sql::col::TYPE,
                work
            ) +
        " LIMIT 1 ";

    return work.exec(query).affected_rows() > 0;
}

} // namespace maps::wiki::social::tasks
