#include "helpers.h"

#include <maps/wikimap/mapspro/libs/social/helpers.h>
#include <maps/wikimap/mapspro/libs/social/magic_strings.h>

#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/social/event_filter.h>

#include <fmt/format.h>

#include <chrono>

namespace maps::wiki::social::tasks {

namespace {

const boost::format SELECT_EDIT_TASKS_BY_COMMIT_IDS(
    "SELECT " + sql::col::EVENT_ID +
    " FROM " + sql::table::TASK_ACTIVE +
    " WHERE " + sql::col::COMMIT_ID + " IN (%1%)"
      " AND " + sql::col::TYPE + "='" + sql::value::EVENT_TYPE_EDIT + "'");

const boost::format SELECT_AND_LOCK_TASKS(
    "SELECT " + sql::col::EVENT_ID + "," + sql::col::RESOLVED_BY +
    " FROM " + sql::table::TASK_ACTIVE +
    " WHERE %1% IN (%2%)"
    " ORDER BY " + sql::col::EVENT_ID + " DESC FOR UPDATE");

const std::string NON_LOCKED =
    sql::col::LOCKED_AT + " IS NULL";

const std::string LOCKED =
    sql::col::LOCKED_AT + " IS NOT NULL";

const std::string LOCK_VALID =
    sql::col::LOCKED_AT + " >= (" + sql::value::NOW + " - " + sql::value::LOCK_DURATION + ")";

const std::string LOCK_EXPIRED =
    sql::col::LOCKED_AT + " < (" + sql::value::NOW + " - " + sql::value::LOCK_DURATION + ")";

LockedTaskIds
selectForUpdateTasks(
    pqxx::transaction_base& work,
    const TIds& ids,
    const std::string& fieldName)
{
    LockedTaskIds result;
    if (ids.empty()) {
        return result;
    }

    boost::format query(SELECT_AND_LOCK_TASKS);
    query % fieldName % common::join(ids, ',');

    for (const auto& row : work.exec(query.str())) {
        auto taskId = row[sql::col::EVENT_ID].as<TId>();
        if (row[sql::col::RESOLVED_BY].as<TUid>() != 0) {
            result.resolved.insert(taskId);
        } else {
            result.notResolved.insert(taskId);
        }
    }
    return result;
}

TaskIds
commitIdsToEditTaskIds(
    pqxx::transaction_base& work,
    const TIds& commitIds)
{
    TaskIds result;
    if (!commitIds.empty()) {
        boost::format query(SELECT_EDIT_TASKS_BY_COMMIT_IDS);
        query % common::join(commitIds, ',');
        for (const auto& row : work.exec(query.str())) {
            result.insert(row[sql::col::EVENT_ID].as<TId>());
        }
    }
    return result;
}

} // namespace

LockedTaskIds
selectForUpdateTasksByTaskIds(
    pqxx::transaction_base& work,
    const TaskIds& taskIds)
{
    return selectForUpdateTasks(work, taskIds, sql::col::EVENT_ID);
}

LockedTaskIds
selectForUpdateEditTasksByCommitIds(
    pqxx::transaction_base& work,
    const TIds& commitIds)
{
    return selectForUpdateTasks(
        work,
        commitIds,
        sql::col::TYPE + "='" + sql::value::EVENT_TYPE_EDIT + "'"
            " AND " + sql::col::COMMIT_ID);
}

TaskIds
commitIdsToEditTaskIds(
    pqxx::transaction_base& work,
    const TIds& commitIds, size_t batchSize)
{
    if (!batchSize) {
        return commitIdsToEditTaskIds(work, commitIds);
    }

    TaskIds taskIds;
    auto it = commitIds.begin();
    for (auto count = commitIds.size(); count; ) {
        auto batchCount = std::min(count, batchSize);
        auto end = it;
        std::advance(end, batchCount);

        auto batchTaskIds = commitIdsToEditTaskIds(work, {it, end});
        taskIds.insert(batchTaskIds.begin(), batchTaskIds.end());

        count -= batchCount;
        it = end;
    }
    return taskIds;
}

TaskIds
loadTaskIds(const pqxx::result& rows)
{
    TaskIds taskIds;
    for (const auto& row : rows) {
        taskIds.insert(row[sql::col::EVENT_ID].as<TId>());
    }
    return taskIds;
}

// categoryIds | commonTasksPermitted | Result
// ------------+----------------------+-------
//        none |            none/true | "" - filtration is not needed at all / all tasks are accessible
//        none |                false | "category_id IS NOT NULL"
// ------------+----------------------+-------
//       empty |            none/true | "category_id IS NULL"
//       empty |                false | error - categories are not set and NULL is prohibited too
// ------------+----------------------+-------
//  categories |            none/true | "category_id IN (...) OR category_id IS NULL"
//  categories |                false | "category_id IN (...)"

std::string
primaryObjectCategoryIdClause(
    const EventFilter& filter,
    const std::string& tableAlias,
    pqxx::transaction_base& work)
{
    bool commonTasksPermitted = !filter.commonTasksPermitted() || *(filter.commonTasksPermitted());

    REQUIRE(
        !(filter.categoryIds() && filter.categoryIds()->empty() && !commonTasksPermitted),
        "Impossible to acquire tasks without access to categories and common tasks"
    );

    if (!filter.categoryIds() && commonTasksPermitted) {
        return {};
    }

    if (!filter.categoryIds() && !commonTasksPermitted) {
        return " AND " + tableAlias + sql::col::PRIMARY_OBJECT_CATEGORY_ID + " IS NOT NULL";
    }

    std::string categoriesClause;
    if (filter.categoryIds() && !filter.categoryIds()->empty()) {
        categoriesClause =
            tableAlias + sql::col::PRIMARY_OBJECT_CATEGORY_ID + " IN ("
            + common::join(
                *filter.categoryIds(),
                [&](const std::string& cat) { return work.quote(cat); },
                ',')
            + ")";
    }

    std::string commonTasksPermittedClause;
    if (commonTasksPermitted) {
        commonTasksPermittedClause = tableAlias + sql::col::PRIMARY_OBJECT_CATEGORY_ID + " IS NULL";
    }

    if (!categoriesClause.empty() && !commonTasksPermittedClause.empty()) {
        return " AND (" + categoriesClause + " OR " + commonTasksPermittedClause + ")";
    }

    return " AND " + categoriesClause + commonTasksPermittedClause;
}


std::string
suspiciousUsersClause(
    const std::optional<bool>& suspiciousUsers,
    const std::string& commitEventTableAlias)
{
    if (!suspiciousUsers) {
        return {};
    }

    return
        " AND " + commitEventTableAlias + sql::col::CREATED_BY + (*suspiciousUsers ? "" : " NOT") + " IN (\n"
        "  SELECT " + sql::col::UID + "\n"
        "  FROM " + sql::table::SUSPICIOUS_USERS + "\n"
        "  WHERE\n"
        "    " + sql::col::FIRST_COMMIT_AT + " < " + sql::col::REGISTERED_OR_UNBANNED_AT + " + " + sql::value::SUSPICIOUS_FIRST_COMMIT_INTERVAL + "\n"
        "    OR " + sql::col::CHANGES_0_30_SEC + " >= " + sql::value::SUSPICIOUS_CHANGES_0_30_SEC + "\n"
        ")\n";
}


std::string noviceUsersClause (
    const std::optional<bool>& noviceUsers,
    const std::string& taskTableAlias)
{
    if (!noviceUsers) {
        return {};
    }

    return " AND " + taskTableAlias + sql::col::IS_CREATED_BY_NOVICE +
        " IS" + (*noviceUsers ? "" : " NOT") + " TRUE";
}


std::string
commitEventTaskClause(
    std::optional<ModerationMode> mode,
    const std::string& taskAlias,
    const ModerationTimeIntervals& moderationTimeIntervals)
{
    using namespace fmt::literals;

    if (!mode) {
        return {};
    }

    switch (*mode) {
        case ModerationMode::Moderator:
            return " AND " + taskAlias + sql::col::RESOLVED_BY + " = 0";
        case ModerationMode::SuperModerator:
        case ModerationMode::Supervisor:
            return fmt::format(
                " AND ("
                    "{task}{resolved_by}  = 0 AND {task}{created_at}  < (NOW() - {create_delay}) OR "
                    "{task}{resolved_by} != 0 AND {task}{resolved_at} < (NOW() - {resolve_delay})"
                ")",
                "task"_a = taskAlias,
                "created_at"_a = sql::col::CREATED_AT,
                "resolved_at"_a = sql::col::RESOLVED_AT,
                "resolved_by"_a = sql::col::RESOLVED_BY,
                "create_delay"_a = sqlIntervalInSeconds(moderationTimeIntervals.superModeratorDelay),
                "resolve_delay"_a = sqlIntervalInSeconds(moderationTimeIntervals.supervisorDelay)
            );
    }
}


std::string
feedbackEventTaskClause(const std::string& taskAlias)
{
    // Only Cartographer can work with feedback event moderation tasks
    //
    return " AND " + taskAlias + sql::col::RESOLVED_BY + "!=0";
}

std::string aoiTaskActiveClause(std::optional<TId> aoiId, const std::string& taskAlias)
{
    if (!aoiId) {
        return {};
    }

    return " AND " + taskAlias + sql::col::EVENT_ID + " IN (" +
        "SELECT " + sql::col::EVENT_ID + " FROM " + sql::table::AOI_FEED_TRUNK_TASK_ACTIVE +
        " WHERE " + sql::col::AOI_ID + "=" + std::to_string(*aoiId) + ")";
}

std::string
nonLockedClause()
{
    return " AND (" + NON_LOCKED + " OR " + LOCK_EXPIRED + ")";
}

std::string
lockedClause(TUid uid)
{
    return " AND (" + sql::col::LOCKED_BY + "=" + std::to_string(uid) +
           " AND " + LOCK_VALID + ")";
}

std::string
oldClause(
    ModerationMode mode,
    const std::string& taskAlias,
    const ModerationTimeIntervals& moderationTimeIntervals)
{
    if (mode == ModerationMode::Moderator) {
        return " AND (" + taskAlias + sql::col::CREATED_AT + " < ("
            + sql::value::NOW + " - "
            + sqlIntervalInSeconds(moderationTimeIntervals.moderatorOldTaskAge) + "))";
    }
    return " AND (" + taskAlias + sql::col::CREATED_AT + " < ("
            + sql::value::NOW + " - "
            + sqlIntervalInSeconds(moderationTimeIntervals.superModeratorOldTaskAge) + "))";
}

std::string
lockedByOthersClause(TUid uid)
{
    return " AND (" + LOCKED + " AND "  + sql::col::LOCKED_BY + "!=" + std::to_string(uid) +
           " AND " + LOCK_VALID + ")";
}

std::string
acquirableClause(TUid uid)
{
    return " AND (" + sql::col::LOCKED_BY + "=" + std::to_string(uid) +
            " OR " + NON_LOCKED + " OR " + LOCK_EXPIRED + ")";
}

std::string
deferredClause(
    std::optional<Deferred> deferred,
    TUid uid,
    const std::string& taskAlias)
{
    if (!deferred) {
        return {};
    }

    return std::string(" AND ") + (*deferred == Deferred::No ? "NOT " : "") + "EXISTS("
                "SELECT " + sql::col::EVENT_ID + " FROM " + sql::table::DEFERRED_TASK + " dt" +
                " WHERE " + sql::col::DEFERRED_BY + "=" + std::to_string(uid) +
                  " AND " + sql::value::NOW + " < " + sql::col::EXPIRES_AT +
                  " AND dt." + sql::col::EVENT_ID + "=" + taskAlias + sql::col::EVENT_ID + ")";
}

size_t
oldTaskAgeInHours(
    ModerationMode mode,
    const ModerationTimeIntervals& moderationTimeIntervals)
{
    return std::chrono::duration_cast<std::chrono::hours>(mode == ModerationMode::Moderator
        ? moderationTimeIntervals.moderatorOldTaskAge
        : moderationTimeIntervals.superModeratorOldTaskAge).count();
}

} // namespace maps::wiki::social::tasks
