#include "users.h"
#include "magic_strings.h"
#include <yandex/maps/wiki/social/feedback/history.h>

#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <sstream>

namespace maps::wiki::social::impl {

namespace {

const std::vector<feedback::TaskOperation> FEEDBACK_TASK_RESOLVE_OPERATIONS {
    feedback::TaskOperation::Accept,
    feedback::TaskOperation::Reject,
};

const auto ACTIVITY_FEEDBACK_RESOLVE_CLAUSE =
    " AND " + sql::col::OPERATION +
    " IN ('" + common::join(FEEDBACK_TASK_RESOLVE_OPERATIONS, "','") + "')";

const auto ACTIVITY_EDITS_CLAUSE =
    " AND " + sql::col::TYPE + "='" + sql::value::EVENT_TYPE_EDIT + "'";

} // namespace

std::optional<UserData>
getUserData(pqxx::transaction_base& txn, TUid uid)
{
    const std::string query =
        " SELECT * FROM " + sql::table::USERDATA +
        " WHERE " + sql::col::UID + " = " + std::to_string(uid);

    auto r = txn.exec(query);
    if (r.empty()) {
        return std::nullopt;
    }
    return UserData(r[0]);
}

UserData
setUserData(pqxx::transaction_base& txn, TUid uid, const std::string& data)
{
    const auto quotedData = txn.quote(data);
    const std::string query =
        "INSERT INTO " + sql::table::USERDATA +
        " (" + sql::col::UID + "," + sql::col::DATA + ") VALUES ("
             + std::to_string(uid) + "," + quotedData + ")"
        " ON CONFLICT (" + sql::col::UID + ")"
        " DO UPDATE SET " +
            sql::col::MODIFIED_AT + "=NOW(), " +
            sql::col::DATA + "=" + quotedData +
        " RETURNING *";

    return UserData(txn.exec1(query));
}

UserActivity getUserActivity(
    pqxx::transaction_base& txn,
    TUid uid,
    const std::vector<std::chrono::seconds>& timeIntervals,
    ActivityType type)
{
    if (timeIntervals.empty()) {
        return {};
    }

    auto uidStr = std::to_string(uid);
    std::vector<std::string> queryParts;
    for (const auto& interval : timeIntervals) {
        auto intervalStr = std::to_string(interval.count());

        auto selectClause = [&](const auto& table, const auto& countColumn) {
            return "SELECT " + intervalStr + ", COUNT(" + countColumn + ") FROM " + table;
        };
        auto whereClause = [&](const auto& columnBy, const auto& columnAt) {
            return
                " WHERE " + columnBy + "=" + uidStr +
                " AND " + columnAt + " > (" + sql::value::NOW +
                    " - INTERVAL '" + intervalStr + " sec')";
        };

        switch (type) {
            case ActivityType::FeedbackResolve:
                queryParts.push_back(
                    selectClause(sql::table::FEEDBACK_HISTORY, "DISTINCT " + sql::col::MODIFIED_AT) +
                    whereClause(sql::col::MODIFIED_BY, sql::col::MODIFIED_AT) +
                    ACTIVITY_FEEDBACK_RESOLVE_CLAUSE
                );
                break;
            case ActivityType::Edits:
                queryParts.push_back(
                    selectClause(sql::table::COMMIT_EVENT, "1") +
                    whereClause(sql::col::CREATED_BY, sql::col::CREATED_AT) +
                    ACTIVITY_EDITS_CLAUSE
                );
                break;
            case ActivityType::Comments:
                queryParts.push_back(
                    selectClause(sql::table::COMMENT, "1") +
                    whereClause(sql::col::CREATED_BY, sql::col::CREATED_AT)
                );
                break;
        }
    }

    social::UserActivity activity;
    auto r = txn.exec(common::join(queryParts, " UNION ALL "));
    for (const auto& row : r) {
        activity.emplace(std::chrono::seconds(row[0].as<size_t>()), row[1].as<size_t>());
    }

    return activity;
}

void saveUserActivity(
    pqxx::transaction_base& txn,
    TUid uid,
    const std::string& ip,
    std::optional<uint16_t> port,
    UserActivityAction action,
    const std::optional<TId>& entityId)
{
    REQUIRE(!ip.empty(), "empty ip address");

    static const auto FIELDS =
        sql::col::UID + "," +
        sql::col::IP_ADDR + "," +
        sql::col::ACTION + "," +
        sql::col::PORT + "," +
        sql::col::ENTITY_ID;

    std::ostringstream query;
    query <<
        "INSERT INTO " << sql::table::USER_ACTIVITY <<
        " (" + FIELDS + ") VALUES (" <<
            uid << "," <<
            txn.quote(ip) << ","
            "'" << action << "',";
    if (port) {
        query << *port << ",";
    } else {
        query << "NULL,";
    }
    if (entityId) {
        query << *entityId;
    } else {
        query << "NULL";
    }
    query << ")";

    txn.exec(query.str());
}

void saveUserActivityAlert(
    pqxx::transaction_base& txn,
    TUid uid,
    const std::string& reason)
{
    txn.exec(
        "INSERT INTO " + sql::table::USER_ACTIVITY_ALERTS +
        "(" + sql::col::UID + "," + sql::col::REASON + ") VALUES (" +
            std::to_string(uid) + "," + txn.quote(reason) + ")");
}

} // namespace maps::wiki::social::impl
