#include <maps/wikimap/mapspro/libs/social_serv_serialize/include/jsonize_assessment.h>

#include <maps/wikimap/mapspro/libs/common/include/yandex/maps/wiki/common/json_helpers.h>

namespace maps::wiki::socialsrv::serialize {

using namespace maps::wiki::assessment;

namespace {

const std::string ID = "id";
const std::string GRADED_BY = "gradedBy";
const std::string GRADED_AT = "gradedAt";
const std::string VALUE = "value";
const std::string COMMENT = "comment";

const std::string ENTITY_ID = "entityId";
const std::string ENTITY_DOMAIN = "entityDomain";
const std::string QUALIFICATION = "qualification";
const std::string ACTION_BY = "actionBy";
const std::string ACTION_AT = "actionAt";
const std::string ACTION = "action";
const std::string GRADES = "grades";
const std::string PERMISSIONS = "permissions";
const std::string FIXABLE = "fixable";
const std::string REFUTATION_ACCEPTABLE = "refutationAcceptable";
const std::string UNITS = "units";

const std::string CREATED_AT = "createdAt";
const std::string NAME = "name";
const std::string GRADED = "graded";
const std::string STATS = "stats";
const std::string TOTAL = "total";

const std::string HAS_MORE = "hasMore";
const std::string SAMPLES = "samples";

const std::string USER_STATS = "userStats";
const std::string UID = "uid";
const std::string RECENT_GRADED_AT = "recentGradedAt";

} // namespace

void
jsonize(json::ObjectBuilder& builder, const Unit& unit)
{
    builder[ID] = common::idToJson(unit.id);
    builder[ENTITY_ID] = unit.entity.id;
    builder[ENTITY_DOMAIN] = toString(unit.entity.domain);
    builder[ACTION_BY] = common::idToJson(unit.action.by);
    builder[ACTION_AT] = chrono::formatIsoDateTime(unit.action.at);
    builder[ACTION] = unit.action.name;
}

void
jsonize(json::ObjectBuilder& builder, const Grade& grade)
{
    builder[ID] = common::idToJson(grade.id);
    if (grade.gradedBy != 0) {
        builder[GRADED_BY] = common::idToJson(grade.gradedBy);
    }
    builder[GRADED_AT] = chrono::formatIsoDateTime(grade.gradedAt);
    builder[VALUE] = toString(grade.value);
    if (grade.comment) {
        builder[COMMENT] = *grade.comment;
    }
    builder[QUALIFICATION] = toString(grade.qualification);
}

namespace {

namespace serialize = maps::wiki::socialsrv::serialize;

void
jsonize(json::ObjectBuilder& builder, const UnitPermissionFlags& permissions)
{
    if (permissions.isSet(UnitPermission::Fixable)) {
        builder[FIXABLE] = true;
    }
    if (permissions.isSet(UnitPermission::RefutationAcceptable)) {
        builder[REFUTATION_ACCEPTABLE] = true;
    }
}

void
jsonize(json::ObjectBuilder& builder, const GradedUnit& gradedUnit)
{
    serialize::jsonize(builder, static_cast<const Unit&>(gradedUnit));
    builder[GRADES] << [&](json::ArrayBuilder builder) {
        for (const auto& grade : gradedUnit.grades) {
            builder << [&](json::ObjectBuilder builder) {
                serialize::jsonize(builder, grade);
            };
        }
    };
    if (gradedUnit.permissions.any()) {
        builder[PERMISSIONS] = [&](json::ObjectBuilder builder) {
            jsonize(builder, gradedUnit.permissions);
        };
    }
}

} // namespace

void
jsonize(json::ArrayBuilder& builder, const GradedUnitVec& gradedUnits)
{
    for (const auto& gradedUnit : gradedUnits) {
        builder << [&](json::ObjectBuilder builder) {
            jsonize(builder, gradedUnit);
        };
    }
}

void
jsonize(json::ObjectBuilder& builder, const UnitFeed& unitFeed)
{
    builder[HAS_MORE] = unitFeed.hasMore();
    builder[UNITS] = [&](json::ArrayBuilder builder) {
        jsonize(builder, unitFeed.units());
    };
}

namespace {

void jsonize(json::ObjectBuilder& builder, const SampleInfo::AssessorStats& assessorStats)
{
    builder[UID] = common::idToJson(assessorStats.uid);
    builder[GRADED] = assessorStats.gradedCount;
    builder[RECENT_GRADED_AT] = chrono::formatIsoDateTime(assessorStats.recentGradedAt);
}

void jsonize(json::ArrayBuilder& builder, const std::vector<SampleInfo::AssessorStats>& assessorsStats)
{
    for (const auto& assessorStats : assessorsStats) {
        builder << [&](json::ObjectBuilder builder) {
            jsonize(builder, assessorStats);
        };
    }
}

} // namespace

void
jsonize(json::ObjectBuilder& builder, const SampleInfo& sampleInfo)
{
    builder[ID] = common::idToJson(sampleInfo.id);
    builder[ENTITY_DOMAIN] = toString(sampleInfo.entityDomain);
    builder[QUALIFICATION] = toString(sampleInfo.qualification);
    builder[CREATED_AT] = chrono::formatIsoDateTime(sampleInfo.createdAt);
    builder[NAME] = sampleInfo.name;
    builder[STATS] << [&](json::ObjectBuilder builder) {
        builder[GRADED] = sampleInfo.gradedCount;
        builder[TOTAL] = sampleInfo.totalCount;
    };
    builder[USER_STATS] = [&](json::ArrayBuilder builder) {
        jsonize(builder, sampleInfo.assessorsStats);
    };
}

namespace {

void
jsonize(json::ArrayBuilder& builder, const SampleInfoVec& samplesInfo)
{
    for (const auto& sampleInfo : samplesInfo) {
        builder << [&](json::ObjectBuilder builder) {
            serialize::jsonize(builder, sampleInfo);
        };
    }
}

} // namespace

void
jsonize(json::ObjectBuilder& builder, const SampleFeed& sampleFeed)
{
    builder[HAS_MORE] = sampleFeed.hasMore();
    builder[SAMPLES] = [&](json::ArrayBuilder builder) {
        jsonize(builder, sampleFeed.samples());
    };
}

} // namespace maps::wiki::socialsrv::serialize
