#include <yandex/maps/wiki/tasks/branch.h>
#include <yandex/maps/wiki/tasks/tasks.h>

#include <yandex/maps/wiki/common/pg_advisory_lock_ids.h>
#include <yandex/maps/wiki/revision/branch_manager.h>

namespace maps {
namespace wiki {
namespace tasks {

namespace {

const std::string REVISION_META_COMMITS_QUEUE_TABLE = "revision_meta.commits_queue";
const std::string REVISION_META_GEOMETRY_TABLE = "revision_meta.geometry";
const std::string BRANCH_ID_COLUMN = "branch_id";
const std::string LABELS_TABLE = "labels";

const revision::Branch::LockId GLOBAL_LOCK_ID = 0;

bool schemaExists(
    pqxx::transaction_base& work,
    const std::string& schemaName)
{
    auto query =
        "SELECT 1 FROM pg_tables"
        " WHERE schemaname = " + work.quote(schemaName) + " LIMIT 1";
    return !work.exec(query).empty();
}

} // namespace

void publishBranch(
    pqxx::transaction_base& workCore,
    pqxx::transaction_base& workLabels,
    const revision::Branch& branchToPublish,
    revision::UserID uid)
{
    revision::BranchManager branchManager(workCore);
    auto branch = branchManager.load(branchToPublish.id()); //reload branch data
    branch.lock(workCore, GLOBAL_LOCK_ID, revision::Branch::LockMode::Nowait);

    auto archiveBranches = branchManager.load({{revision::BranchType::Archive, 1}});

    REQUIRE(branch.state() == revision::BranchState::Normal,
        "Stable branch is not in normal state");
    REQUIRE(branch.finishedBy() == 0, "Branch is already finished");

    auto result = branch.finish(workCore, uid);
    REQUIRE(result, "Failed to finish branch");

    if (!archiveBranches.empty()) {
        auto archiveBranchId = archiveBranches.front().id();
        auto schemaName = "vrevisions_stable_" + std::to_string(archiveBranchId);
        if (schemaExists(workLabels, schemaName)) {
            workLabels.exec("TRUNCATE " + schemaName + "." + LABELS_TABLE + " CASCADE");
        }
    }
}

void deleteBranch(
    pqxx::transaction_base& workCore,
    pqxx::transaction_base& workView,
    const revision::Branch& branchToDelete,
    revision::UserID uid)
{
    auto branchType = branchToDelete.type();
    REQUIRE(branchType != revision::BranchType::Trunk, "Can not delete trunk branch");
    REQUIRE(branchType != revision::BranchType::Approved, "Can not delete approved branch");
    REQUIRE(branchType != revision::BranchType::Deleted, "Can not delete already deleted branch");

    revision::BranchManager branchManager(workCore);
    auto branch = branchManager.load(branchToDelete.id()); //reload branch data
    branch.lock(workCore, GLOBAL_LOCK_ID, revision::Branch::LockMode::Nowait);

    REQUIRE(branch.state() != revision::BranchState::Progress,
        "Branch " << branch.id() << " is in progress state.");

    std::ostringstream queryLockViews;
    queryLockViews << "SELECT pg_advisory_xact_lock ";
    queryLockViews << "(" << static_cast<int64_t>(common::AdvisoryLockIds::SYNC_VIEW) << ")";
    workView.exec(queryLockViews.str());

    if (branchType == revision::BranchType::Stable) {
        branch.finish(workCore, uid);
    }

    branch.setState(workCore, revision::BranchState::Unavailable);
    branch.setType(workCore, revision::BranchType::Deleted);

    std::ostringstream queryDeleteQueue;
    queryDeleteQueue << "DELETE FROM " << REVISION_META_COMMITS_QUEUE_TABLE;
    queryDeleteQueue << " WHERE " << BRANCH_ID_COLUMN << " = " << branch.id();
    workCore.exec(queryDeleteQueue.str());

    std::ostringstream queryDeleteBboxes;
    queryDeleteBboxes << "DELETE FROM " << REVISION_META_GEOMETRY_TABLE;
    queryDeleteBboxes << " WHERE " << BRANCH_ID_COLUMN << " = " << branch.id();
    workCore.exec(queryDeleteBboxes.str());

    std::ostringstream queryDeleteVRevisionsData;
    queryDeleteVRevisionsData << "SELECT vrevisions_stable.delete_branch(" << branch.id() << ")";
    workView.exec(queryDeleteVRevisionsData.str());

    deleteVrevisionsForBranch(workView, branchToDelete);
}

void deleteVrevisionsForBranch(
    pqxx::transaction_base& work,
    const revision::Branch& branchToDelete)
{
    std::ostringstream queryDeleteVRevisionsSchema;
    queryDeleteVRevisionsSchema << "DROP SCHEMA IF EXISTS";
    queryDeleteVRevisionsSchema << " vrevisions_stable_" << branchToDelete.id() << " CASCADE";
    work.exec(queryDeleteVRevisionsSchema.str());
}

} // namespace tasks
} // namespace wiki
} // namespace maps
