#pragma once

#include <maps/libs/common/include/exception.h>
#include <maps/libs/pgpool/include/pgpool3.h>

#include <boost/optional.hpp>

#include <string>
#include <vector>

namespace maps {
namespace wiki {
namespace tasks {

typedef uint64_t TaskId;
typedef uint64_t CommitId;

enum class CommitStatus
{
    NotSynced = false,
    Synced = true
};

struct CommitRecord
{
    TaskId taskId;
    CommitId commitId;
    CommitStatus status;

    CommitRecord(TaskId tId, CommitId cId, CommitStatus s)
        : taskId(tId), commitId(cId), status(s)
    {}
};

typedef std::vector<CommitRecord> CommitRecords;

struct CommitRecordNotFound : public maps::Exception
{
    CommitRecordNotFound(TaskId taskId_, CommitId commitId_)
        : maps::Exception("Not found commit for task id " +
            std::to_string(taskId_) + ", commit id " + std::to_string(commitId_))
        , taskId(taskId_), commitId(commitId_)
    {
    }
    const TaskId taskId;
    const CommitId commitId;
};

struct CommitRecordInvalid : public maps::Exception
{
    CommitRecordInvalid() : maps::Exception()
    {}
    explicit CommitRecordInvalid(const std::string& what)
        : maps::Exception(what)
    {}
};

/**
 * Gateway to operate with commit synchronization records
 * The table is expected to have following columns:
 *   task_id (bigint), commit_id (bigint), synchronized (boolean)
 *   PRIMARY KEY (task_id, commit_id)
 */
class CommitRecordGateway
{
public:
    CommitRecordGateway(pgpool3::TransactionHandle& txn, std::string tableName);

    CommitRecord getCommit(TaskId taskId, CommitId commitId) const;

    template<typename CommitIds>
    CommitRecords getCommits(TaskId taskId, CommitIds&& commitIds) const;

    /// @return all commits for a task
    CommitRecords getCommits(TaskId taskId) const;

    /// @returns all commit IDs for a task
    template<typename CommitIds = std::vector<CommitId>>
    CommitIds getCommitIds(TaskId taskId) const;

    void addCommit(const CommitRecord& entry);

    template<typename CommitIds>
    void addCommits(TaskId taskId, CommitIds&& commitIds, CommitStatus status);

    /// @return true if a commit entry was removed, false otherwise
    bool removeCommit(TaskId taskId, CommitId commitId);

    /// @return number of affected rows
    template<typename CommitIds>
    std::size_t removeCommits(TaskId taskId, CommitIds&& commitIds);

    /// Removes all commits for a task
    /// @return number of affected rows
    std::size_t removeCommits(TaskId taskId);

    /// @throws CommitRecordNotFound if affected taskId/commitId does not exist
    void updateStatus(
        TaskId taskId,
        CommitId commitId,
        CommitStatus status);

    /// @return number of affected rows
    template<typename CommitIds>
    std::size_t updateStatus(
        TaskId taskId,
        CommitIds&& commitIds,
        CommitStatus status);

private:
    pgpool3::TransactionHandle& txn_;
    std::string tableName_;

    template<typename CommitIds>
    CommitRecords getCommitsInternal(
        TaskId taskId,
        const boost::optional<CommitIds>& commitIds) const;
};

std::ostream& operator<<(std::ostream& out, CommitStatus status);

} // tasks
} // namespace wiki
} // namespace maps

#include "commit_sync.inl"

