#include <yandex/maps/wiki/tasks/revocation_checker.h>

#include <yandex/maps/wiki/tasks/tasks.h>

#include <maps/libs/concurrent/include/scoped_guard.h>
#include <maps/libs/log8/include/log8.h>

namespace maps::wiki::tasks {

namespace {
    constexpr auto VALUE_DIRTY_TIME = std::chrono::seconds(10);
} // anonymous namespace

RevocationChecker::RevocationChecker(pgpool3::Pool& pool, TaskId taskId)
    : pool_(pool)
    , taskId_(taskId)
{
}

bool RevocationChecker::isTaskRevoked()
{
    if (lastValue_.load(std::memory_order_relaxed)) {
        // terminated
        return true;
    }

    update();

    // we don't care if the value is actually updated even if it is dirty
    return lastValue_.load(std::memory_order_relaxed);
}

void RevocationChecker::update()
{
    if (isUpdating_.test_and_set(std::memory_order_acquire)) {
        return;
    }

    maps::concurrent::ScopedGuard guard([&] {
        isUpdating_.clear(std::memory_order_release);
    });

    if (Clock::now() < nextUpdateTime_) {
        return;
    }

    try {
        auto maybeStatus = getOverriddenStatus(pool_, taskId_);
        lastValue_.store(
            maybeStatus && (*maybeStatus == TaskStatus::Revoked),
            std::memory_order_relaxed
        );
    } catch (const TaskNotFoundException& e) {
        throw;
    } catch (const maps::Exception& e) {
        WARN() << e;
    } catch (const std::exception& e) {
        WARN() << e.what();
    }

    // finally (not in case of TaskNotFoundException) - update timestamp
    nextUpdateTime_ = Clock::now() + VALUE_DIRTY_TIME;
}

} // namespace maps::wiki::tasks
