#include <yandex/maps/wiki/tasks/task_logger.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/common/retry_duration.h>

#include <ctime>
#include <sstream>

namespace maps::wiki::tasks {

namespace sql {
namespace tbl {
    const std::string TASK = "service.task";
} // namespace tbl
namespace col {
    const std::string LOG = "log";
    const std::string ID = "id";
} // namespace col
} // namespace sql

namespace {

void printTimestamp(std::ostream& os)
{
    static const char* format = "[%d/%m/%Y %H:%M:%S]";
    constexpr size_t MAX_LEN = 40;
    char buffer[MAX_LEN];

    time_t rawtime;
    ::time(&rawtime);
    auto ret = ::strftime(buffer, MAX_LEN, format, ::localtime(&rawtime));
    REQUIRE(ret > 0, "Failed to format timestamp");
    os << buffer;
}

std::string severityToStr(Severity severity)
{
    switch (severity) {
    case Severity::Info:
        return "INFO";
    case Severity::Warn:
        return "WARN";
    case Severity::Error:
        return "ERROR";
    }
    throw maps::LogicError()
            << "Invalid severity: " << static_cast<int>(severity);
}

} // namespace

std::ostream& operator<< (std::ostream& os, Severity severity)
{
    return os << severityToStr(severity);
}

//=================== Message ===================

Message::Message(TaskPgLogger& logger, Severity severity)
    : severity_(severity)
    , logger_(logger)
{
    printTimestamp(buf_);
    buf_ << " " << severity_ << ": ";
}

Message::~Message()
{
    logger_.log(buf_.str());
}

//=================== TaskPgLogger ===================

TaskPgLogger::TaskPgLogger(
        maps::pgpool3::Pool& pgPool,
        TaskId taskId)
    : pgPool_(pgPool)
    , taskId_(taskId)
{
    logInfo() << "Starting task id: " << taskId;
}

Message TaskPgLogger::logInfo() {
    return Message(*this, Severity::Info);
}

Message TaskPgLogger::logWarn() {
    return Message(*this, Severity::Warn);
}

Message TaskPgLogger::logError() {
    return Message(*this, Severity::Error);
}

void TaskPgLogger::log(const std::string& str) try
{
    common::retryDuration([&] {
        auto txn = pgPool_.masterWriteableTransaction();

        std::ostringstream query;
        query << "UPDATE " << sql::tbl::TASK << "\n"
              << "SET " << sql::col::LOG << " = " << sql::col::LOG
                        << " || " << txn->quote(str + "\n") << "\n"
              << "WHERE " << sql::col::ID << " = " << taskId_;

        txn->exec(query.str());
        txn->commit();
    });
} catch (const std::exception& e) {
    WARN() << "Unable to log task progress to Postgres. "
           << "Message: \'" << str << "\'. "
           << "Exception: " << e.what();
}

} // namespace maps::wiki::tasks
