#include <yandex/maps/wiki/tasks/task_manager.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <maps/libs/http/include/http.h>

#include <maps/libs/log8/include/log8.h>

#include <chrono>
#include <set>
#include <thread>

namespace maps {
namespace wiki {
namespace tasks {

namespace {

constexpr int HTTP_STATUS_OK = 200;

constexpr auto POLLING_INTERVAL = std::chrono::seconds(5);

std::string performRequest(
    const http::Method& method,
    const std::string& url,
    const std::string& body = std::string{},
    ContentType contentType = ContentType::Form)
{
    http::URL urlWrapper(url);
    http::Client httpClient;
    http::Request request(httpClient, method, urlWrapper);
    if (!body.empty()) {
        switch (contentType) {
        case ContentType::Form:
            request.addHeader("Content-Type", "application/x-www-form-urlencoded");
            break;
        case ContentType::Xml:
            request.addHeader("Content-Type", "application/xml");
            break;
        }
        request.setContent(body);
    }
    auto response = request.perform();
    if (response.status() != HTTP_STATUS_OK) {
        throw ResponseError{}
            << "status: " << response.status() << "; "
            << "method: " << method << "; "
            << "url: " << url;
    }
    return response.readBody();
}

TaskInfo performRequestAndGetTaskInfo(
    const http::Method& method,
    const std::string& url,
    const std::string& body = std::string{},
    ContentType contentType = ContentType::Form)
{
    try {
        return TaskInfo::fromXml(performRequest(method, url, body, contentType));
    } catch (const maps::Exception& e) {
        ERROR() << "Failed to start task with request '" << url << "' and body '" << body << "'";
        throw;
    }
}

} // namespace

TaskManager::TaskManager(std::string tasksUrl, UserId uid)
    : tasksUrl_(std::move(tasksUrl))
    , uid_(uid)
{}

TaskInfo TaskManager::startCreateStableBranch(BranchId approvedBranchId) const
{
    std::ostringstream url;
    url << tasksUrl_ << "tasks"
        << "?uid=" << uid_
        << "&type=vrevisions_refresh"
        << "&action=create-stable"
        << "&branch=" << approvedBranchId;
    return performRequestAndGetTaskInfo(http::POST, url.str());
}

TaskInfo TaskManager::startApplyShadowAttributes(BranchId stableBranchId) const
{
    std::ostringstream url;
    url << tasksUrl_ << "tasks"
        << "?uid=" << uid_
        << "&type=apply_shadow_attributes"
        << "&branch=" << stableBranchId;
    return performRequestAndGetTaskInfo(http::POST, url.str());
}

TaskInfo TaskManager::startValidation(
    BranchId branchId,
    PresetId presetId,
    boost::optional<ObjectId> aoiId,
    boost::optional<ObjectId> regionId,
    boost::optional<CommitId> commitId) const
{
    REQUIRE (!aoiId || !regionId, "Aoi and region can't be set simultaneously");
    std::ostringstream url;
    url << tasksUrl_ << "tasks"
        << "?uid=" << uid_
        << "&type=validation"
        << "&branch=" << branchId
        << "&preset=" << presetId;
    if (aoiId) {
        url << "&aoi=" << *aoiId;
    }
    if (regionId) {
        url << "&region=" << *regionId;
    }
    if (commitId) {
        url << "&commit=" << *commitId;
    }
    return performRequestAndGetTaskInfo(http::POST, url.str());
}

TaskInfo TaskManager::startValidation(
    BranchId branchId,
    const std::set<std::string>& checks,
    boost::optional<ObjectId> aoiId,
    boost::optional<ObjectId> regionId,
    boost::optional<CommitId> commitId,
    boost::optional<TaskId> parentId) const
{
    REQUIRE (!aoiId || !regionId, "Aoi and region can't be set simultaneously");
    std::ostringstream url;
    url << tasksUrl_ << "tasks"
        << "?uid=" << uid_
        << "&type=validation"
        << "&branch=" << branchId;
    if (aoiId) {
        url << "&aoi=" << *aoiId;
    }
    if (regionId) {
        url << "&region=" << *regionId;
    }
    if (commitId) {
        url << "&commit=" << *commitId;
    }
    if (parentId) {
        url << "&parent=" << *parentId;
    }
    auto body = "checks=" + common::join(checks, ',');
    return performRequestAndGetTaskInfo(http::POST, url.str(), body, ContentType::Form);
}

TaskInfo TaskManager::startDiffalert(
    BranchId oldBranchId,
    BranchId newBranchId) const
{
    std::ostringstream url;
    url << tasksUrl_ << "tasks"
        << "?uid=" << uid_
        << "&type=diffalert"
        << "&base-branch=" << oldBranchId
        << "&branch=" << newBranchId;
    return performRequestAndGetTaskInfo(http::POST, url.str());
}

TaskInfo TaskManager::startExport(
    BranchId stableBranchId,
    const std::string& subset,
    ExportTested tested,
    boost::optional<CommitId> commitId) const
{
    std::ostringstream url;
    url << tasksUrl_ << "tasks"
        << "?uid=" << uid_
        << "&type=export"
        << "&branch=" << stableBranchId
        << "&subset=" << subset
        << "&tested=" << (tested == ExportTested::Yes ? "1" : "0");
    if (commitId) {
        url << "&commit=" << *commitId;
    }
    return performRequestAndGetTaskInfo(http::POST, url.str());
}

TaskInfo TaskManager::taskInfo(
    TaskId taskId,
    const Token& token,
    maps::common::RetryPolicy retryPolicy) const
{
    return maps::common::retry(
        [&]() {
            std::ostringstream url;
            url << tasksUrl_ << "tasks/" << taskId;
            if (!token.empty()) {
                url << "?token=" << token;
            }
            return TaskInfo::fromXml(performRequest(http::GET, url.str()));
        },
        retryPolicy
    );
}

TaskInfo TaskManager::resumeTask(TaskId taskId) const
{
    std::ostringstream url;
    url << tasksUrl_ << "tasks/" << taskId
        << "?uid=" << uid_
        << "&status=pending";
    return TaskInfo::fromXml(performRequest(http::PUT, url.str()));
}

void TaskManager::changeTaskParameters(TaskId taskId, const std::string& body) const
{
    std::ostringstream url;
    url << tasksUrl_ << "tasks/" << taskId << "/parameters"
        << "?uid=" << uid_;
    performRequest(http::PUT, url.str(), body, ContentType::Xml);
}

boost::optional<TaskStatus> TaskManager::getTaskStatus(const TaskInfo& task) const
{
    try {
        auto updatedTaskInfo = taskInfo(task.id(), task.token());
        return updatedTaskInfo.status();
    } catch (const ResponseError& e) {
        WARN() << e;
    } catch (const maps::http::Error& e) {
        WARN() << e;
    } catch (const std::exception& e) {
        WARN() << e.what();
    }
    return boost::none;
}

TaskManager::TaskWaitResult TaskManager::waitForTasks(
    TaskInfos taskInfos,
    std::chrono::seconds duration,
    CheckCancelCallback checkCancel) const
{
    TaskWaitResult result;

    auto startTimePoint = std::chrono::system_clock::now();
    auto endTimePoint = startTimePoint + duration;

    while (!taskInfos.empty() && std::chrono::system_clock::now() < endTimePoint) {
        checkCancel();
        taskInfos.remove_if([&](const TaskInfo& task) {
            auto taskStatus = getTaskStatus(task);
            if (taskStatus && *taskStatus != TaskStatus::InProgress) {
                result.taskStatuses.emplace(task.id(), *taskStatus);
                return true;
            }
            return false;
        });
        std::this_thread::sleep_for(POLLING_INTERVAL);
    }

    for (const auto& task : taskInfos) {
        result.taskStatuses.emplace(task.id(), TaskStatus::Failed);
    }

    return result;
}

bool TaskManager::TaskWaitResult::isSuccess(TaskId taskId) const
{
    auto it = taskStatuses.find(taskId);
    return it != taskStatuses.end() && it->second == TaskStatus::Success;
}

} // namespace tasks
} // namespace wiki
} // namespace maps
