#include <maps/wikimap/mapspro/libs/taskutils/impl/tasks_impl.h>

#include <maps/wikimap/mapspro/libs/taskutils/include/exception.h>
#include <maps/wikimap/mapspro/libs/taskutils/include/token.h>

#include <map>

namespace maps::wiki::taskutils {

namespace {

class StatusResolver {
public:
    StatusResolver();

    TaskStatus resolve(const std::string& str) const;

    const std::string& resolve(TaskStatus status) const;

private:
    typedef std::map<std::string, TaskStatus> StatusMap;
    typedef std::map<TaskStatus, std::string> ReverseMap;

    StatusMap data_;
    ReverseMap reverseData_;
};

const StatusResolver g_statusResolver;


StatusResolver::StatusResolver()
{
    data_["created"]    = TaskStatus::Created;
    data_["started"]    = TaskStatus::Started;
    data_["done"]       = TaskStatus::Done;
    data_["failed"]     = TaskStatus::Failed;
    data_["canceled"]   = TaskStatus::Canceled;
    data_["terminated"] = TaskStatus::Terminated;

    for (auto it = data_.begin(), end = data_.end(); it != end; ++it) {
        reverseData_[it->second] = it->first;
    }
    reverseData_[TaskStatus::Expired] = "expired";
    reverseData_[TaskStatus::Unknown] = "unknown";
}

TaskStatus
StatusResolver::resolve(const std::string& str) const
{
    auto it = data_.find(str);
    if (it == data_.end()) {
        throw UnknownStatusException() << "unknown task status string: " << str;
    }
    return it->second;
}

const std::string&
StatusResolver::resolve(TaskStatus status) const
{
    auto it = reverseData_.find(status);
    if (it == reverseData_.end()) {
        throw UnknownStatusException() << "unknown task status: " << status;
    }
    return it->second;
}

} // namespace


TasksCommonData::TasksCommonData(const std::string& dbSchema)
    : dbSchema_(dbSchema)
{
}

void
TasksCommonData::registerTask(const Token& token)
{
    std::lock_guard<std::mutex> guard(mutex_);
    ids_.insert(token.id());
}

void
TasksCommonData::unregisterTask(const Token& token)
{
    std::lock_guard<std::mutex> guard(mutex_);
    ids_.erase(token.id());
}

void
TasksCommonData::swapTasks(std::set<TaskID>& ids)
{
    std::lock_guard<std::mutex> guard(mutex_);
    ids_.swap(ids);
}


TaskResultImpl::TaskResultImpl(TaskStatus status, const std::string &data)
    : status_(status)
{
    if (status == TaskStatus::Done || status == TaskStatus::Failed) {
        data_ = data;
    }
}

TaskResultImpl::TaskResultImpl(
    const std::string& status, const std::string &result, const std::string &errMsg)
    : status_(g_statusResolver.resolve(status))
{
    if (status_ == TaskStatus::Done) {
        data_ = result;
    }
    else if (status_ == TaskStatus::Failed) {
        data_ = errMsg;
    }
}

const std::string&
TaskResultImpl::statusStr() const
{
    return g_statusResolver.resolve(status_);
}


TaskImpl::TaskImpl(std::shared_ptr<TasksCommonData> commonData, const TaskInfo& info, const Token& token)
    : commonData_(commonData)
    , info_(info)
    , token_(token)
{
    ASSERT(commonData_);
    commonData_->registerTask(token_);
}

TaskImpl::~TaskImpl()
{
    commonData_->unregisterTask(token_);
}

} // namespace maps::wiki::taskutils
