#include "cache_impl.h"

#include "graph.h"
#include "node_impl.h"
#include "edge_impl.h"
#include "editor_impl.h"
#include "path_utils.h"
#include "face_validator_impl.h"

#include <yandex/maps/wiki/topo/storage.h>
#include <yandex/maps/wiki/topo/editor.h>
#include <yandex/maps/wiki/topo/exception.h>
#include <yandex/maps/wiki/topo/face_validator.h>

#include <list>

namespace maps {
namespace wiki {
namespace topo {

CacheImpl::CacheImpl(Storage& storage, double tolerance,
        std::unique_ptr<index::SpatialIndex> index)
    : storage_(storage)
    , graph_(new Graph(std::move(index), tolerance))
{}

IncidencesByNodeMap CacheImpl::incidencesByNodes(const NodeIDSet& nodeIds)
{
    loadByNodes(nodeIds);

    IncidencesByNodeMap res;

    NodeConstIteratorRange nodeRange = graph_->nodes();
    for (auto it = nodeRange.begin(); it != nodeRange.end(); ++it) {
        if (nodeIds.find(it->id()) != nodeIds.end()) {
            IncidentEdges& edges = res[it->id()];
            for (const auto& edgePtr : it->impl_->startIncidentEdges()) {
                edges.emplace_back(edgePtr->id(), IncidenceType::Start);
            }
            for (const auto& edgePtr : it->impl_->endIncidentEdges()) {
                edges.emplace_back(edgePtr->id(), IncidenceType::End);
            }
            typedef IncidentEdges::value_type IncEdgePair;
            std::sort(
                edges.begin(), edges.end(),
                [] (const IncEdgePair& lhs, const IncEdgePair& rhs) -> bool
                {
                    return lhs.first < rhs.first;
                }
            );
        }
    }
    return res;
}

IncidencesByEdgeMap CacheImpl::incidencesByEdges(const EdgeIDSet& edgeIds)
{
    loadByEdges(edgeIds);

    IncidencesByEdgeMap res;

    EdgeConstIteratorRange edgeRange = graph_->edges();
    for (auto it = edgeRange.begin(); it != edgeRange.end(); ++it) {
        if (edgeIds.find(it->id()) != edgeIds.end()) {
            res[it->id()] = IncidentNodes(it->startNode(), it->endNode());
        }
    }
    return res;
}

boost::optional<Circuit> CacheImpl::circuit(const EdgeIDSet& edgeIds, bool withNodes)
{
    if (edgeIds.empty()) {
        return boost::none;
    }

    loadByEdges(edgeIds);

    auto paths = buildPaths(incidencesByEdges(edgeIds));
    ASSERT(!paths.empty());
    if (paths.size() > 1) {
        return boost::none;
    }

    const Path& path = paths.front();
    if (path.isClosed()) {
        return boost::none;
    }

    return withNodes
        ? Circuit {
            EdgeIDVector(path.edgeIds.begin(), path.edgeIds.end()),
            NodeIDVector(path.nodeIds.begin(), path.nodeIds.end())}
        : Circuit {
            EdgeIDVector(path.edgeIds.begin(), path.edgeIds.end()),
            boost::none};
}

void CacheImpl::loadObjects(const geolib3::BoundingBox& bbox)
{
    EdgeIDSet edgeIds = storage_.edgeIds(bbox);
    EdgeVector edges = storage_.edges(edgeIds);
    IncidencesByEdgeMap incidences = storage_.incidencesByEdges(edgeIds);
    NodeIDSet nodesToLoad;
    for (const auto& incPair : incidences) {
        nodesToLoad.insert(incPair.second.start);
        nodesToLoad.insert(incPair.second.end);
    }
    NodeIDSet nodesByBBox = storage_.nodeIds(bbox);
    nodesToLoad.insert(nodesByBBox.begin(), nodesByBBox.end());
    NodeVector nodes = storage_.nodes(nodesToLoad);
    // nodes must be loaded first
    for (auto& node : nodes) {
        graph_->addNode(node);
    }
    for (auto& edge : edges) {
        graph_->addEdge(edge);
    }
}

void CacheImpl::loadObjects(const geolib3::Point2& center,
    double width, double height)
{
    loadObjects(geolib3::BoundingBox(center, width, height));
}

void CacheImpl::loadObjects(const geolib3::Polyline2& polyline,
    double withinDistance)
{
    const geolib3::BoundingBox& bbox = polyline.boundingBox();
    return loadObjects(
        bbox.center(),
        bbox.width() + 2 * withinDistance,
        bbox.height() + 2 * withinDistance);
}

void CacheImpl::loadByNodes(const NodeIDSet& nodeIds)
{
    NodeIDSet nodesToLoad;
    for (auto nodeId : nodeIds) {
        if (!graph_->nodeExists(nodeId) ||
            !graph_->node(nodeId).impl_->areIncidentEdgesLoaded())
        {
            nodesToLoad.insert(nodeId);
        }
    }
    IncidencesByNodeMap incidences = storage_.incidencesByNodes(nodeIds);
    EdgeIDSet edgesToLoad;
    for (const auto& incidencePair : incidences) {
        for (const auto& edgeInfo : incidencePair.second) {
            edgesToLoad.insert(edgeInfo.first);
        }
    }
    EdgeVector edges = storage_.edges(edgesToLoad);
    for (const auto& edge : edges) {
        nodesToLoad.insert(edge.startNode());
        nodesToLoad.insert(edge.endNode());
    }
    NodeVector nodes = storage_.nodes(nodesToLoad);
    // nodes must be loaded first
    for (auto& node : nodes) {
        node.impl_->setIncidentEdgesLoaded();
        graph_->addNode(node);
    }
    for (auto& edge : edges) {
        graph_->addEdge(edge);
    }
}

void CacheImpl::loadByEdges(const EdgeIDSet& edgeIds)
{
    EdgeVector edges = storage_.edges(edgeIds);
    NodeIDSet nodesToLoad;
    for (const auto& edge : edges) {
        nodesToLoad.insert(edge.startNode());
        nodesToLoad.insert(edge.endNode());
    }
    NodeVector nodes = storage_.nodes(nodesToLoad);
    // nodes must be loaded first
    for (auto& node : nodes) {
        graph_->addNode(node);
    }
    for (auto& edge : edges) {
        graph_->addEdge(edge);
    }
}

void CacheImpl::loadEdgeFaces(const EdgeIDSet& edgeIds)
{
    EdgeIDSet edgesToLoad;
    for (auto id : edgeIds) {
        if (!graph_->edgeExists(id)) {
            edgesToLoad.insert(id);
        }
    }
    loadByEdges(edgesToLoad);
    for (const auto& edgeFaces : storage_.facesByEdges(edgeIds)) {
        if (!graph_->edgeExists(edgeFaces.first)) {
            continue;
        }
        graph_->edge(edgeFaces.first).impl_->faceIds = edgeFaces.second;
    }
}

std::unique_ptr<Editor> CacheImpl::editor()
{
    return std::make_unique<Editor>(std::make_unique<EditorImpl>(*this));
}

std::unique_ptr<FaceValidator> CacheImpl::faceValidator()
{
    return std::unique_ptr<FaceValidator>(new FaceValidator(new FaceValidator::Impl(*this)));
}

} // namespace topo
} // namespace wiki
} // namespace maps
