#include "create_intersections.h"

#include "save_edge_helpers.h"
#include "preload_objects.h"
#include "process_events.h"
#include "../index/spatial_index.h"
#include "../geom_tools/intersector.h"
#include "../events_data.h"
#include "../editor_impl.h"
#include "../cache_impl.h"
#include "../graph.h"

#include <yandex/maps/wiki/topo/storage.h>
#include <yandex/maps/wiki/topo/events.h>
#include <yandex/maps/wiki/topo/exception.h>

#include <maps/libs/geolib/include/intersection.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/vector.h>

#include <vector>
#include <algorithm>

namespace maps {
namespace wiki {
namespace topo {

namespace {

const EdgeID FAKE_NEW_EDGE_ID = 1;

void
updateSplitEdgeIds(SplitEdgeIdsMap& allSplitEdges, const SplitEdgeIdsMap& partialSplitEdges)
{
    if (partialSplitEdges.empty()) {
        return;
    }

    for (const auto& ids : partialSplitEdges) {
        const auto id = ids.first;
        const auto& partIds = ids.second;
        bool isPartOfSplitElement = false;
        for (auto& elementPartIds : allSplitEdges) {
            auto& elementParts = elementPartIds.second;
            auto elementIt = elementParts.find(id);
            if (elementIt == elementParts.end()) {
                continue;
            }
            elementParts.erase(elementIt);
            elementParts.insert(partIds.begin(), partIds.end());
            isPartOfSplitElement = true;
            break;
        }
        if (!isPartOfSplitElement) {
            allSplitEdges.insert(ids);
        }
    }
}

/// Can only touch at ends
void
checkNoIncorrectIntersections(const geolib3::PolylinesVector& geoms)
{
    auto isEndpoint = [] (const geolib3::Point2& p, const geolib3::Polyline2& geom)
    {
        return p == geom.points().front() || p == geom.points().back();
    };

    for (size_t i = 0; i < geoms.size(); ++i) {
        for (size_t j = i + 1; j < geoms.size(); ++j) {
            const auto& g1 = geoms[i];
            const auto& g2 = geoms[j];
            auto intersection = geolib3::intersection(g1, g2);
            if (intersection.empty()) {
                continue;
            }
            const auto& intersectionPoint = intersection.front().points().front();
            if (intersection.size() > 1 || intersection.front().points().size() != 1) {
                throw GeomProcessingErrorWithLocation(
                        ErrorCode::UnexpectedIntersection, intersectionPoint)
                    << " geoms for CreateIntersections can only touch at endpoints";
            }

            if (!isEndpoint(intersectionPoint, g1) || !isEndpoint(intersectionPoint, g2)) {
                throw GeomProcessingErrorWithLocation(
                        ErrorCode::UnexpectedIntersection, intersectionPoint)
                    << " geoms for CreateIntersections can only touch at endpoints";
            }
        }
    }
}

} // namespace

CreateIntersectionsOperation::CreateIntersectionsOperation(
        const Callbacks& callbacks,
        CacheImpl& cache,
        const geolib3::PolylinesVector& geoms,
        const TopologyRestrictions& restrictions)
    : Operation(callbacks, cache)
    , geoms_(geoms)
    , restrictions_(restrictions)
{}

Editor::CreateIntersectionsResult
CreateIntersectionsOperation::operator () ()
{
    for (auto& geom : geoms_) {
        geom = prepareGeometry(geom);
    }

    checkNoIncorrectIntersections(geoms_);
    preloadObjects(cache_, TopologyUpdateData(geoms_), restrictions_);

    Editor::CreateIntersectionsResult result;

    ProcessEvents process(cache_, callbacks_);

    for (const auto& geom : geoms_) {
        // compute intersections
        SourceEdgeID sourceId = {FAKE_NEW_EDGE_ID, /*bool exists =*/ false};
        geom::Intersector intersector(
            cache_.graph(), restrictions_,
            [&] (NodeID /*nodeId*/) -> bool { return false; });
        geom::Intersector::Events  events  = intersector(sourceId, geom);

        auto splitEdgeIds = processSplittedEdges(process, events);
        sendSplitEdgeEventOfIntersectedEdges(callbacks_, events);
        deleteUnusedNodes(cache_, events.unusedNodeIds);

        const SplitDataPtr& geomData = events.editedEdgeEvent;
        ASSERT(!geomData->splitPoints.empty());
        geolib3::PointsVector adjustedGeomPoints;
        for (const auto& splitPolyline: geomData->splitPolylines) {
            const auto& points = splitPolyline->geom.points();
            ASSERT(points.size() >= 2);
            adjustedGeomPoints.insert(
                adjustedGeomPoints.end(),
                points.begin(), std::prev(points.end()));
        }
        adjustedGeomPoints.push_back(geomData->splitPolylines.back()->geom.points().back());

        updateSplitEdgeIds(result.splitEdgeIds, splitEdgeIds);
        result.adjustedGeoms.push_back(geolib3::Polyline2(adjustedGeomPoints));
    }

    return result;
}

} // namespace topo
} // namespace wiki
} // namespace maps
