#include "move_node.h"

#include "save_edge.h"
#include "process_events.h"
#include "preload_objects.h"
#include "../geom_tools/make_event.h"
#include "../geom_tools/util.h"
#include "../editor_impl.h"
#include "../cache_impl.h"
#include "../graph.h"
#include "../events_data.h"
#include "../index/spatial_index.h"

#include <yandex/maps/wiki/topo/storage.h>
#include <yandex/maps/wiki/topo/editor.h>
#include <yandex/maps/wiki/topo/events.h>
#include <yandex/maps/wiki/topo/common.h>
#include <yandex/maps/wiki/topo/exception.h>

#include <maps/libs/geolib/include/closest_point.h>
#include <maps/libs/geolib/include/distance.h>


namespace maps {
namespace wiki {
namespace topo {


MoveNodeOperation::MoveNodeOperation(
        const Callbacks& callbacks,
        CacheImpl& cache,
        const Editor::NodeData& data,
        const TopologyRestrictions& restrictions)
    : Operation(callbacks, cache)
    , nodeId_(data.id)
    , pos_(data.pos)
    , restrictions_(restrictions)
{}

void MoveNodeOperation::operator () ()
{
    TopologyUpdateData updateData{
        {{nodeId_, &pos_}},
        {}
    };
    preloadObjects(cache_, updateData, restrictions_);

    moveNodeOnly();

    for (auto edgePtr : cache_.graph().incidentEdges(nodeId_)) {
        Editor::EdgeData data{
            SourceEdgeID(edgePtr->id(), true),
            geolib3::PointsVector(),
            edgePtr->geom()
        };

        SaveEdgeOperation{
            callbacks_, cache_, data, restrictions_
        }(/*bool loadObjects = */false);
    }
}

void MoveNodeOperation::moveNodeOnly()
{
    const NodeIDVector nearestNodeIds = cache_.graph().index().nearestNodes(
        pos_, restrictions_.junctionGravity(), {nodeId_}
    );
    if (nearestNodeIds.size() > 1) {
        throw GeomProcessingErrorWithLocation(ErrorCode::AmbiguousSnapping, pos_)
            <<  " node " << nodeId_ << " snaps to "
                " several closest nodes, two closest ids " << nearestNodeIds[0]
            <<  ", " << nearestNodeIds[1];
    } else if (nearestNodeIds.size() == 1) {
        const Node& nearestNode = cache_.graph().node(nearestNodeIds[0]);
        MergeNodesEventData mergeData(
            nodeId_, nearestNode.id(), nearestNode.pos()
        );
        ProcessEvents(cache_, callbacks_)(mergeData);
        nodeId_ = mergeData.idTo;
        pos_ = mergeData.pos;
    } else {
        const NodeIDVector ignoredEdgeIds = cache_.graph().node(nodeId_).incidentEdgeIds();
        const EdgeIDVector nearestEdgeIds = cache_.graph().index().nearestEdges(
            pos_, restrictions_.junctionGravity(),
            {ignoredEdgeIds.begin(), ignoredEdgeIds.end()}
        );
        if (nearestEdgeIds.size() > 1) {
            throw GeomProcessingErrorWithLocation(ErrorCode::AmbiguousSnapping, pos_)
                << " node " << nodeId_ << " snaps to "
                   " several closest edges, two closest ids " << nearestEdgeIds[0]
                << ", " << nearestEdgeIds[1];
        } else if (nearestEdgeIds.size() == 1) {
            snapEdgeSplit(cache_.graph().edge(nearestEdgeIds[0]));
        } else {
            processNodeMove();
        }
    }
}

void MoveNodeOperation::processNodeMove()
{
    cache_.graph().moveNode(nodeId_, pos_);
    cache_.storage().updateNodePos(nodeId_, pos_);
    EdgePtrVector incidentEdges = cache_.graph().incidentEdges(nodeId_);
    for (auto edgePtr : incidentEdges) {
        cache_.storage().updateEdgeGeom(
            edgePtr->id(), edgePtr->geom(),
            IncidentNodes(edgePtr->startNode(), edgePtr->endNode())
        );
    }
}

void MoveNodeOperation::snapEdgeSplit(const Edge& snapEdge)
{
    REQUIRE(!snapEdge.isIncident(nodeId_),
        "Node " << nodeId_ << " is moved on edge " <<
        snapEdge.id() << " which is already incident to this node"
    );

    const geolib3::Polyline2& polyline = snapEdge.geom();

    pos_ = geolib3::closestPoint(polyline, pos_);
    const size_t vertexIndex = polyline.closestNodeIndex(pos_);
    const geom::SnapPoint snapPoint =
        geom::near(polyline.pointAt(vertexIndex), pos_, restrictions_.junctionGravity())
        ? geom::SnapPoint{snapEdge, vertexIndex}
        : geom::SnapPoint{snapEdge, polyline.closestPointSegmentIndex(pos_), pos_};
    pos_ = snapPoint.geom();

    const OptionalNodeID resnapNodeId = cache_.graph().index().nodeAt(
        pos_, restrictions_.junctionGravity(), {nodeId_}
    );

    if (resnapNodeId) {
        throw GeomProcessingErrorWithLocation(ErrorCode::AmbiguousSnapping, pos_)
            <<  " node " << nodeId_ << " is snapped to "
                "other node " << *resnapNodeId << " after snapping to edge " << snapEdge.id();
    }

    checkRestrictions(cache_.graph().node(nodeId_));
    geom::SnapPolylineVector splits;

    splits.emplace_back(
        geom::SnapPointVector {
            geom::SnapPoint { cache_.graph().node(snapEdge.startNode()) }
        }
    );

    splits.emplace_back(geom::SnapPointVector{snapPoint});

    splits.emplace_back(
        geom::SnapPointVector {
            geom::SnapPoint { cache_.graph().node(snapEdge.endNode()) }
        }
    );

    geom::SplitEdgeEventMaker maker;
    geom::SplitEdgeEventDataPtr splitData = maker.makeForIntersectedEdge(
        snapEdge, splits
    );

    REQUIRE(
        splitData->splitPoints.size() == 3,
        "SplitEdgeEventMaker must create 3 split points. There is some error."
    );

    processNodeMove();
    splitData->splitPoints[1]->nodeId = nodeId_;    // set node id for split point

    ProcessEvents(cache_, callbacks_)(*splitData);
    callbacks_.splitEdge.callback().processEvent(
        SplitEdgeEvent{*splitData}
    );
}


void MoveNodeOperation::checkRestrictions(const Node& node) const
{
    if (restrictions_.maxIntersectedEdges() &&
        *restrictions_.maxIntersectedEdges() < node.degree())
    {
        throw GeomProcessingError(ErrorCode::TooManyIntersectionsWithNetwork) <<
            "Too many edges intersected";
    }
    if (restrictions_.maxIntersectionsWithEdge() &&
        *restrictions_.maxIntersectionsWithEdge() == 0)
    {
        throw GeomProcessingError(ErrorCode::TooManyIntersectionsWithElement) <<
            "Too many intersections with one edge";
    }
}

} // namespace topo
} // namespace wiki
} // namespace maps
