#include "save_edge_helpers.h"

#include "../graph.h"
#include "../events_data.h"
#include "../geom_tools/validation.h"
#include "../editor_impl.h"

#include <yandex/maps/wiki/topo/events.h>
#include <yandex/maps/wiki/topo/editor.h>
#include <yandex/maps/wiki/topo/storage.h>
#include <yandex/maps/wiki/topo/exception.h>

#include <algorithm>
#include <iterator>


namespace maps {
namespace wiki {
namespace topo {

geolib3::Polyline2
prepareGeometry(const geolib3::Polyline2& geom)
{
    // remove degenerate segments
    auto simplifiedGeom = geolib3::unique(geom);
    return simplifiedGeom;
}

void
preloadObjectsByGeometry(
    CacheImpl& cache,
    OptionalEdgeID id, const geolib3::Polyline2& geom,
    const TopologyRestrictions& restrictions)
{
    auto bbox = geom.boundingBox();
    if (id) {
        cache.loadByEdges( EdgeIDSet {*id} );
        const Edge& edge = cache.graph().edge(*id);
        bbox = geolib3::expand(edge.geom().boundingBox(), bbox);
    }
    bbox = geolib3::resizeByValue(bbox, Editor::maxInteractionDistance(restrictions));
    cache.loadObjects(bbox);
}

void
deleteUnusedNodes(CacheImpl& cache, const NodeIDSet& unusedNodeIds)
{
    for (auto nodeId: unusedNodeIds) {
        const Node& node = cache.graph().node(nodeId);
        if (node.isIsolated() && cache.storage().nodeCanBeDeleted(nodeId)) {
            cache.storage().deleteNode(nodeId);
            cache.graph().deleteNode(nodeId);
        }
    }
}

SplitEdgeIdsMap
processSplittedEdges(ProcessEvents& process, geom::Intersector::Events& events)
{
    SplitEdgeIdsMap splitEdgeIds;
    for (auto& pair: events.intersectedEdgesEvents) {
        SplitDataPtr& eventData = pair.second;
        REQUIRE(
            eventData->sourceId.exists(),
            "Existing edge source must exist, id " << eventData->sourceId.id()
        );
        process(*eventData);
        const SplitPolylinePtrVector& splitPolylines = eventData->splitPolylines;
        if (splitPolylines.size() > 1) {
            EdgeIDSet edgeIdSet;
            for (const auto& splitPolyline: splitPolylines) {
                edgeIdSet.insert(*splitPolyline->edgeId);
            }
            splitEdgeIds.insert({pair.first, edgeIdSet});
        }
    }
    return splitEdgeIds;
}

void
deleteMergedEdges(ProcessEvents& process, CacheImpl& cache, geom::Intersector::Events& events)
{
    auto deleteEdgeIfNeed = [&process, &cache](const SplitEdgeEventData& event)
    {
        if (!event.partToKeepIDIndex) {
            const EdgeID& id = event.sourceId.id();
            const Edge& edge = cache.graph().edge(id);
            DeleteEdgeEventData deleteEventData{id, edge.startNode(), edge.endNode()};
            process(deleteEventData);
        }
    };

    deleteEdgeIfNeed(*events.editedEdgeEvent);
    for (const auto& pair: events.intersectedEdgesEvents) {
        deleteEdgeIfNeed(*pair.second);
    }
}

void
sendSplitEdgeEventOfIntersectedEdges(const Callbacks& callbacks, geom::Intersector::Events& events)
{
    for (const auto& pair: events.intersectedEdgesEvents) {
        callbacks.splitEdge.callback().processEvent(
            SplitEdgeEvent{*pair.second}
        );
    }
}


void
processEdgesSaving(CacheImpl& cache, const Callbacks& callbacks, geom::Intersector::Events& events)
{
    ProcessEvents process(cache, callbacks);

    process(*events.editedEdgeEvent);
    processSplittedEdges(process, events);

    if (events.editedEdgeEvent->splitPolylines.size() > 1 ||
            events.editedEdgeEvent->splitPolylines[0]->isShared)
    {
        callbacks.splitEdge.callback().processEvent(
            SplitEdgeEvent{*events.editedEdgeEvent}
        );
    }
    sendSplitEdgeEventOfIntersectedEdges(callbacks, events);

    deleteMergedEdges(process, cache, events);
    deleteUnusedNodes(cache, events.unusedNodeIds);
}

} // namespace topo
} // namespace wiki
} // namespace maps
