#pragma once

#include <yandex/maps/wiki/topo/editor.h>
#include <yandex/maps/wiki/topo/exception.h>

#include <boost/noncopyable.hpp>

namespace maps {
namespace wiki {
namespace topo {

class CacheImpl;
struct Callbacks;

class EditorImpl : public boost::noncopyable {
public:
    explicit EditorImpl(CacheImpl& cacheImpl);

    void saveEdge(
        const Editor::EdgeData& data,
        const TopologyRestrictions& restrictions);

    void saveNode(
        const Editor::NodeData& data,
        const TopologyRestrictions& restrictions);

    void saveObjects(
        const Editor::TopologyData& data,
        const TopologyRestrictions& restrictions);

    Editor::CreateIntersectionsResult
    createIntersections(
        const geolib3::PolylinesVector& geoms,
        const TopologyRestrictions& restrictions);

    void deleteEdge(EdgeID id);
    void mergeEdges(NodeID commonNodeId);

    void snapNodes(
        const NodeIDSet& nodeIds,
        const TopologyRestrictions& restrictions);

    inline Callbacks& callbacks() { return *callbacks_; }

private:
    CacheImpl& cacheImpl_;
    std::unique_ptr<Callbacks> callbacks_;
};

template <class Callback>
class CallbackHolder {
public:
    void registerCallback(std::unique_ptr<Callback> callback)
    {
        callback_ = std::move(callback);
    }
    const Callback& callback() const
    {
        REQUIRE(callback_.get(), "Callback not set");
        return *callback_;
    }

    bool isSet() const { return callback_.get() != nullptr; }

private:
    std::unique_ptr<Callback> callback_;
};

struct Callbacks {
    CallbackHolder<TopoAddEdgeCallback> addEdgeTopo;
    CallbackHolder<TopoMoveEdgeCallback> moveEdgeTopo;
    CallbackHolder<TopoDeleteEdgeCallback> deleteEdgeTopo;

    CallbackHolder<SplitEdgeCallback> splitEdge;
    CallbackHolder<MergeEdgesCallback> mergeEdges;
    CallbackHolder<DeleteEdgeCallback> deleteEdge;
    CallbackHolder<MergeNodesCallback> mergeNodes;
};

} // namespace maps::wiki::topo
} // namespace maps::wiki
} // namespace maps
