#include <yandex/maps/wiki/topo/events.h>

#include "events_data.h"

#include <yandex/maps/wiki/topo/exception.h>

namespace maps {
namespace wiki {
namespace topo {

AddEdgeEvent::AddEdgeEvent(const AddEdgeEventData& data)
    : data_(data)
{}

EdgeID AddEdgeEvent::id() const
{
    return data_.id;
}

EdgeID AddEdgeEvent::sourceId() const
{
    return data_.sourceId.id();
}

bool AddEdgeEvent::sourceExists() const
{
    return data_.sourceId.exists();
}

NodeID AddEdgeEvent::startNode() const
{
    return data_.start;
}

NodeID AddEdgeEvent::endNode() const
{
    return data_.end;
}

const geolib3::Polyline2& AddEdgeEvent::geom() const
{
    return data_.geom;
}


DeleteEdgeRequest::DeleteEdgeRequest(DeleteEdgeEventData& data)
    : data_(data)
{}

EdgeID DeleteEdgeRequest::id() const
{
    return data_.id;
}

NodeID DeleteEdgeRequest::startNodeId() const
{
    return data_.startNodeId;
}

NodeID DeleteEdgeRequest::endNodeId() const
{
    return data_.endNodeId;
}


DeleteEdgeEvent::DeleteEdgeEvent(const DeleteEdgeEventData& data)
    : data_(data)
{}

EdgeID DeleteEdgeEvent::id() const
{
    return data_.id;
}

NodeID DeleteEdgeEvent::startNodeId() const
{
    return data_.startNodeId;
}

NodeID DeleteEdgeEvent::endNodeId() const
{
    return data_.endNodeId;
}


SplitEdgeRequest::SplitEdgeRequest(SplitEdgeEventData& data)
    : data_(data)
{}

EdgeID SplitEdgeRequest::sourceId() const
{
    return data_.sourceId.id();
}

bool SplitEdgeRequest::sourceExists() const
{
    return data_.sourceId.exists();
}

bool SplitEdgeRequest::isEditedEdge() const
{
    return data_.isEditedEdge;
}

const SplitPointPtrVector& SplitEdgeRequest::splitPoints() const
{
    return data_.splitPoints;
}

const SplitPolylinePtrVector& SplitEdgeRequest::splitPolylines() const
{
    return data_.splitPolylines;
}

void SplitEdgeRequest::selectPartToKeepID(size_t partNum)
{
    REQUIRE(
        partNum < data_.splitPolylines.size(),
        " split part index out of range: " << partNum
    );
    REQUIRE(
        !data_.splitPolylines[partNum]->edgeId,
        " split part index is already reserved: " << partNum
    );
    data_.partToKeepIDIndex = partNum;
}

const OptionalIndex& SplitEdgeRequest::partToKeepID() const
{
    return data_.partToKeepIDIndex;
}


SplitEdgeEvent::SplitEdgeEvent(const SplitEdgeEventData& data)
    : data_(data)
{}

EdgeID SplitEdgeEvent::sourceId() const
{
    return data_.sourceId.id();
}

bool SplitEdgeEvent::sourceExists() const
{
    return data_.sourceId.exists();
}

EdgeIDVector SplitEdgeEvent::edgeIds() const
{
    EdgeIDVector result;
    result.resize(data_.splitPolylines.size());
    for (size_t i = 0; i < data_.splitPolylines.size(); ++i) {
        REQUIRE(
            data_.splitPolylines[i]->edgeId,
            "Edge id is not set in split edge event for " << i << "-th part"
        );
        result[i] = *data_.splitPolylines[i]->edgeId;
    }
    return result;
}

NodeIDVector SplitEdgeEvent::nodeIds() const
{
    NodeIDVector res;
    res.resize(data_.splitPoints.size());
    for (size_t i = 0; i < data_.splitPoints.size(); ++i) {
        REQUIRE(data_.splitPoints[i]->nodeId, "Node id is not set in split edge event");
        res[i] = *data_.splitPoints[i]->nodeId;
    }
    return res;
}

const OptionalIndex& SplitEdgeEvent::partToKeepID() const
{
    return data_.partToKeepIDIndex;
}


MergeEdgesRequest::MergeEdgesRequest(MergeEdgesEventData& data)
    : data_(data)
{}

EdgeID MergeEdgesRequest::mergedId() const
{
    return data_.mergedId;
}

EdgeID MergeEdgesRequest::deletedId() const
{
    return data_.deletedId;
}

NodeID MergeEdgesRequest::nodeId() const
{
    return data_.mergeNodeId;
}

geolib3::Polyline2 MergeEdgesRequest::newGeom() const
{
    return geolib3::Polyline2(data_.newGeomPoints);
}

void MergeEdgesRequest::swapMergedAndDeleted()
{
    std::swap(data_.mergedId, data_.deletedId);
    if (!data_.isGeomDirectionContinuous) {
        std::reverse(data_.newGeomPoints.begin(), data_.newGeomPoints.end());
    }
}

MergeEdgesEvent::MergeEdgesEvent(const MergeEdgesEventData& data)
    : data_(data)
{}

EdgeID MergeEdgesEvent::mergedId() const
{
    return data_.mergedId;
}

EdgeID MergeEdgesEvent::deletedId() const
{
    return data_.deletedId;
}

NodeID MergeEdgesEvent::nodeId() const
{
    return data_.mergeNodeId;
}

MoveEdgeEvent::MoveEdgeEvent(const MoveEdgeEventData& data)
    : data_(data)
{}

EdgeID MoveEdgeEvent::id() const
{
    return data_.id;
}

OptionalNodeID MoveEdgeEvent::newStartNode() const
{
    return data_.newStartNode;
}

OptionalNodeID MoveEdgeEvent::newEndNode() const
{
    return data_.newEndNode;
}

const geolib3::Polyline2& MoveEdgeEvent::newGeom() const
{
    return data_.newGeom;
}

NodeID MoveEdgeEvent::oldStartNode() const
{
    return data_.oldIncidentNodes.start;
}

NodeID MoveEdgeEvent::oldEndNode() const
{
    return data_.oldIncidentNodes.end;
}


MergeNodesRequest::MergeNodesRequest(MergeNodesEventData& data)
    : data_(data)
{}

NodeID MergeNodesRequest::mergedId() const
{
    return data_.idTo;
}

NodeID MergeNodesRequest::deletedId() const
{
    return data_.idFrom;
}

const geolib3::Point2& MergeNodesRequest::pos() const
{
    return data_.pos;
}

void MergeNodesRequest::swapMergedAndDeleted()
{
    std::swap(data_.idFrom, data_.idTo);
}

MergeNodesEvent::MergeNodesEvent(const MergeNodesEventData& data)
    : data_(data)
{}

NodeID MergeNodesEvent::mergedId() const
{
    return data_.idTo;
}

NodeID MergeNodesEvent::deletedId() const
{
    return data_.idFrom;
}

} // namespace maps::wiki::topo
} // namespace maps::wiki
} // namespace maps
