#include "face_validator_impl.h"

#include "path_utils.h"
#include "graph.h"

#include <yandex/maps/wiki/topo/face_validator.h>
#include <yandex/maps/wiki/topo/storage.h>
#include <yandex/maps/wiki/topo/exception.h>

namespace maps {
namespace wiki {
namespace topo {

namespace {

/**
 * Requires that there is no closed paths both in added and removed lists
 */
void
checkPaths(
    FaceID faceId,
    const PathsList& addedPaths, const PathsList& removedPaths)
{
    typedef std::set<std::pair<NodeID, NodeID>> PathNodesSet;

    auto buildPathNodes = [faceId] (const PathsList& paths)
    {
        PathNodesSet pathNodes;
        for (const auto& path : paths) {
            auto startId = std::min(path.startNodeId, path.endNodeId);
            auto endId = std::max(path.startNodeId, path.endNodeId);
            if (!pathNodes.insert({startId, endId}).second) {
                throw InvalidFaceError(faceId, {startId, endId}) <<
                    "Face " << faceId << " has more than one path between "
                        << startId << " and " <<  endId;
            }
        }
        return pathNodes;
    };

    auto addedPathNodes = buildPathNodes(addedPaths);
    auto removedPathNodes = buildPathNodes(removedPaths);

    auto checkPathNodes = [faceId] (
        const PathNodesSet& pathsNodes, const PathNodesSet& otherPathsNodes,
        const std::string& message)
    {
        for (const auto& nodes : pathsNodes) {
            if (!otherPathsNodes.count(nodes)) {
                throw InvalidFaceError(faceId, {nodes.first, nodes.second}) <<
                    message << ", between " << nodes.first << " and " <<  nodes.second;
            }
        }
    };

    checkPathNodes(addedPathNodes, removedPathNodes,
        "Face " + std::to_string(faceId) + " added path has no match in removed");
    checkPathNodes(removedPathNodes, addedPathNodes,
        "Face " + std::to_string(faceId) + " removed path has no match in added");
}

void
checkNodeDegrees(FaceID faceId, const IncidencesByEdgeMap& incidences)
{
    const IncidencesSetByNodeMap& nodeInc =
        incidencesByEdgesToIncidencesSetByNodes(incidences);
    for (const auto& inc : nodeInc) {
        if (inc.second.size() > 2) {
            throw InvalidFaceError(faceId, {inc.first}) <<
                "Node " << inc.first << " degree within face is > 2";
        }
    }
}

} // namespace

void
FaceValidator::Impl::checkAffectedNodes(
    FaceID faceId, const PathsList& addedPaths)
{
    NodeIDSet nodeIds;
    for (const auto& path : addedPaths) {
        nodeIds.insert(path.nodeIds.begin(), path.nodeIds.end());
    }

    cache_.loadByNodes(nodeIds);

    EdgeIDSet edgeIds;
    for (auto nodeId : nodeIds) {
        for (const auto& edgePtr : cache_.graph().node(nodeId).incidentEdges()) {
            edgeIds.insert(edgePtr->id());
        }
    }

    EdgeIDSet faceEdgeIds;
    cache_.loadEdgeFaces(edgeIds);
    for (auto edgeId : edgeIds) {
        if (cache_.graph().edge(edgeId).isPartOfFace(faceId)) {
            faceEdgeIds.insert(edgeId);
        }
    }

    cache_.loadByEdges(faceEdgeIds);

    auto nodeIncidencesMap = incidencesByEdgesToIncidencesSetByNodes(
        cache_.incidencesByEdges(faceEdgeIds));

    for (const auto& nodeIncidences : nodeIncidencesMap) {
        if (!nodeIds.count(nodeIncidences.first)) {
            continue;
        }
        if (nodeIncidences.second.size() != 2) {
            throw InvalidFaceError(faceId, {nodeIncidences.first}) <<
                "Node " << nodeIncidences.first << " is bound to "
                    << nodeIncidences.second.size() << " elements within face " << faceId;
        }
    }
}

void
FaceValidator::Impl::validateCompleteFace(
    FaceID faceId, const EdgeIDSet& edgeIds)
{
    auto paths = buildPaths(cache_.incidencesByEdges(edgeIds));
    if (paths.size() != 1) {
        throw InvalidFaceError(faceId, {}) <<
            "Face " << faceId << " consists of two or more components";
    }
    if (!paths.front().isClosed()) {
        throw InvalidFaceError(faceId, {}) <<
            "Face " << faceId << " consists of unclosed path";
    }
}

FaceValidator::Impl::FaceDiffPaths
FaceValidator::Impl::buildFaceDiffPaths(FaceID faceId)
{
    auto edgeIdsDiff = cache_.storage().faceDiff(faceId);

    EdgeIDSet affectedEdgeIds = edgeIdsDiff.added;
    affectedEdgeIds.insert(edgeIdsDiff.changed.begin(), edgeIdsDiff.changed.end());

    cache_.loadByEdges(affectedEdgeIds);

    auto addedIncidences = cache_.incidencesByEdges(edgeIdsDiff.added);
    auto movedIncidencesCurrent = cache_.incidencesByEdges(edgeIdsDiff.changed);
    auto movedIncidencesOriginal =
        cache_.storage().originalIncidencesByEdges(edgeIdsDiff.changed);
    auto removedIncidences =
        cache_.storage().originalIncidencesByEdges(edgeIdsDiff.removed);

    addedIncidences.insert(
        movedIncidencesCurrent.begin(), movedIncidencesCurrent.end());
    removedIncidences.insert(
        movedIncidencesOriginal.begin(), movedIncidencesOriginal.end());

    checkNodeDegrees(faceId, addedIncidences);
    checkNodeDegrees(faceId, removedIncidences);

    return {buildPaths(addedIncidences), buildPaths(removedIncidences)};
}

void
FaceValidator::Impl::operator () (FaceID faceId)
{
    auto faceEdgeIds = cache_.storage().tryGetFaceEdges(faceId);
    if (faceEdgeIds) {
        validateCompleteFace(faceId, *faceEdgeIds);
        return;
    }

    auto paths = buildFaceDiffPaths(faceId);

    auto closedPred = [] (const Path& path) { return path.isClosed(); };
    bool hasAddedClosedPath = std::any_of(
        paths.added.begin(), paths.added.end(), closedPred);
    bool hasRemovedClosedPath = std::any_of(
        paths.removed.begin(), paths.removed.end(), closedPred);

    if (hasAddedClosedPath || hasRemovedClosedPath) {
        if (paths.added.size() != 1) {
            throw InvalidFaceError(faceId, {}) <<
                "Face " << faceId << " must have one added loop";
        }
        if (paths.removed.size() != 1) {
            throw InvalidFaceError(faceId, {}) <<
                "Face " << faceId << " must have one removed loop";
        }
        validateCompleteFace(faceId, cache_.storage().getFaceEdges(faceId));
        return;
    }

    checkPaths(faceId, paths.added, paths.removed);
    checkAffectedNodes(faceId, paths.added);
}

// FaceValidator

FaceValidator::FaceValidator(Impl* impl)
    : impl_(impl)
{}

FaceValidator::~FaceValidator() { /* impl */ }

void FaceValidator::operator () (FaceID faceId)
{
    (*impl_)(faceId);
}

} // namespace topo
} // namespace wiki
} // namespace maps
