#pragma once

#include "gen_doc/styles.h"

#include "test_types/mock_storage.h"

#include <yandex/maps/geotex/common.h>
#include <yandex/maps/geotex/objects/point.h>
#include <yandex/maps/geotex/objects/circle.h>
#include <yandex/maps/geotex/objects/polyline.h>
#include <yandex/maps/geotex/objects/segment_buffer.h>

namespace maps {
namespace wiki {
namespace topo {
namespace doc {

class GeotexConvertor {
public:
    // point objects

    geotex::ObjectPtr
    operator () (const geolib3::Point2& point) const
    {
        return this->convert(point, boost::none);
    }

    geotex::ObjectPtr
    operator () (const test::Node& node) const
    {
        return this->convert(node.pos, node.id);
    }

    // polyline objects

    geotex::ObjectPtr
    operator () (const geolib3::Segment2& segment) const
    {
        return this->convert(segment, boost::none);
    }

    geotex::ObjectPtr
    operator () (const geolib3::Polyline2& polyline) const
    {
        return this->convert(polyline, boost::none);
    }

    geotex::ObjectPtr
    operator () (const test::Edge& edge) const
    {
        return this->convert(edge.geom, edge.id);
    }

protected:
    virtual geotex::ObjectPtr
    convert(const geolib3::Point2& point, boost::optional<NodeID> nodeId) const = 0;

    virtual geotex::ObjectPtr
    convert(const geolib3::Polyline2& polyline, boost::optional<EdgeID> edgeId) const = 0;

    virtual geotex::ObjectPtr
    convert(const geolib3::Segment2& segment, boost::optional<EdgeID> edgeId) const = 0;
};


class GeomGeotexConvertor : public GeotexConvertor {
protected:
    virtual geotex::ObjectPtr
    convert(const geolib3::Point2& point, boost::optional<NodeID> nodeId) const
    {
        return nodeId
            ? std::make_shared<geotex::Point>(
                  point,
                  geotex::Label{std::to_string(*nodeId), style::NODE_LABELS, {0.6, 0.6}})
            : std::make_shared<geotex::Point>(point);
    }

    virtual geotex::ObjectPtr
    convert(const geolib3::Segment2& segment, boost::optional<EdgeID> edgeId) const
    {
        return convert(geolib3::Polyline2(segment), edgeId);
    }

    virtual geotex::ObjectPtr
    convert(const geolib3::Polyline2& polyline, boost::optional<EdgeID> edgeId) const
    {
        return edgeId
            ? std::make_shared<geotex::Polyline>(
                  polyline,
                  geotex::Label{std::to_string(*edgeId), style::EDGE_LABELS, {0.6, 0.6}})
            : std::make_shared<geotex::Polyline>(polyline);
    }
};


class GravityGeotexConvertor : public GeotexConvertor {
public:
    explicit GravityGeotexConvertor(double width)
        : width_(width)
    {}

protected:
    virtual geotex::ObjectPtr
    convert(const geolib3::Point2& point, boost::optional<NodeID> /*nodeId*/) const
    {
        return std::make_shared<geotex::Circle>(point, width_);
    }

    virtual geotex::ObjectPtr
    convert(const geolib3::Segment2& segment, boost::optional<EdgeID> /*edgeId*/) const
    {
        return std::make_shared<geotex::SegmentBuffer>(segment, width_);
    }

    virtual geotex::ObjectPtr
    convert(const geolib3::Polyline2& /*polyline*/, boost::optional<EdgeID> /*edgeId*/) const
    {
        REQUIRE(false, "Polyline gravity not implemented");
    }

private:
    double width_;
};

} // namespace doc
} // namespace topo
} // namespace wiki
} // namespace maps
