#include "make_event.h"

#include "geom_io.h"
#include "util.h"

#include <utility>


namespace maps {
namespace wiki {
namespace topo {
namespace geom {


using namespace geolib3;


namespace {

// Copy vertices of polyline from start to end.
// Boundary points are allways included.
PointsVector getSubPoints(const SnapPolyline& polyline, size_t start, const SnapPoint& end)
{
    PointsVector points;

    ASSERT(start < polyline.size());
    if (end == polyline.end()) {
        for (; start < polyline.size(); ++start) {
            points.push_back(polyline[start].geom());
        }
    } else {
        while (polyline[start] != end) {
            points.push_back(polyline[start].geom());
            ++start;
            //This assert arises when two elements or more overlap
            //which is not supported in many ways.
            ASSERT(start < polyline.size());
        }
        points.push_back(end.geom());
    }
    return points;
}

// Copy vertices of polyline from start to end.
// Bound points are allways included, if start = end return one point only.
PointsVector getSubPoints(const Polyline2 polyline, const SnapPoint& start, const SnapPoint& end)
{
    const PointsVector& points = polyline.points();
    PointsVector result;

    auto begin = points.begin();
    if (start.vertexIndex()) {
        begin += *start.vertexIndex();
    } else if (start.segmentIndex()) {
        begin +=  *start.segmentIndex() + 1;
        result.push_back(start.geom());
    }

    if (end.node()) {
        result.insert(result.end(), begin,  points.end());
    } else if (end.vertexIndex()) {
        result.insert(result.end(), begin,  points.begin() + *end.vertexIndex() + 1);
    } else if (end.segmentIndex()) {
        result.insert(result.end(), begin, points.begin() + *end.segmentIndex() + 1);
        if (start != end) { // possible if start is on segment, so start has been pushed already
            result.push_back(end.geom());
        }
    }

    return result;
}

void checkSnapToEdge(const SnapPolyline& split, const Edge& edge)
{
    for (const auto& point: split.points()) {
        if (point.node()) {
            REQUIRE(
                edge.isIncident(*point.node()),
                "Not consistent snap node id " << *point.node()
                    << ", possible ids: " << edge.startNode()
                    << " or " << edge.endNode()
            );
        } else if(point.edge()) {
            REQUIRE(
                point.edge() == edge.id(),
                "Not consistent snap edge id " << *point.edge()
                    << ", " << edge.id() << " expected"
            );
        } else {
            throw RuntimeError() << "Snap point has no any snap to edge " << edge.id();
        }
    }
}

} // end of anonymous namespace


SplitEdgeEventDataPtr SplitEdgeEventMaker::makeForEditedEdge(const SourceEdgeID& sourceEdgeId,
    const SnapPolyline& polyline, const SnapPolylineVector& splits)
{
    SnapPointVector points;
    points.reserve(splits.size());

    // Prepare split points and collect new points for edited polyline.
    // Namely, split, which is polyline, can't give new points due to edited polyline alignment.
    // However, split, which is a single point, can change geometry of edge.
    // It may be intersection of two segments,
    // in this case point can be not added in aligned polyline yet.
    SplitPointPtrVector splitPoints;
    splitPoints.reserve(2 * splits.size());
    for (const auto& split: splits) {
        SnapPoint start = split.start();
        if (split.isPoint()) {
            points.push_back(start);
            splitPoints.push_back(cache(start));
        } else {
             // It is possible that subpolyline is not collinear edited polyline.
            SnapPoint end = split.end();
            if (!polyline.hasSameOrder(split)) {
                std::swap(start, end);
            }
            // if split before is not point than start point is already pushed
            const SplitPointPtr splitPoint = cache(start);
            if (splitPoints.empty() || splitPoints.back() != splitPoint) {
                splitPoints.push_back(splitPoint);
            }
            splitPoints.push_back(cache(end));
        }
    }

    // Add new points to polyline
    const SnapPolyline polylineWithIntersections = supplementPolylineWithPoints(polyline, points);

    // Prepare split polylines
    SplitPolylinePtrVector splitPolylines;
    splitPolylines.reserve(2 * splits.size());
    size_t start = 0;
    for (size_t i = 0; i < splits.size(); ++i) {
        const SnapPolyline& split  = splits[i];

        if (!split.isPoint()) {
            splitPolylines.push_back(cache(split));
            start += split.size() - 1;
        }

        if (i + 1 < splits.size()) { // has next
            const SnapPolyline& nextSplit = splits[i + 1];
            const SnapPoint& end = (nextSplit.isPoint() || polyline.hasSameOrder(nextSplit))
                ? nextSplit.start()
                : nextSplit.end();

            const PointsVector points = getSubPoints(polylineWithIntersections, start, end);
            if (points.size() > 1) {
                splitPolylines.push_back(
                    std::make_shared<SplitPolyline>(Polyline2{points})
                );
                start += points.size() - 1;
            }
        }
    }

    return std::make_unique<SplitEdgeEventData>(
            sourceEdgeId,
            /* isEditedEdge = */ true,
            polyline.geom(),
            splitPoints,
            splitPolylines
    );
}

SplitEdgeEventDataPtr SplitEdgeEventMaker::makeForIntersectedEdge(
    const Edge& edge, const SnapPolylineVector& splits)
{
    SplitPointPtrVector splitPoints;
    SplitPolylinePtrVector splitPolylines;

    splitPoints.reserve(2 * splits.size());
    splitPolylines.reserve(2 * splits.size());

    for (size_t i = 0; i < splits.size(); ++i) {
        const SnapPolyline& split = splits[i];

        // We check only snapping to edge, but not any indices.
        // Indices validity must be guranteed by snap points.
        checkSnapToEdge(split, edge);
        const SplitPointPtr splitPoint = cache(split.start());
        if (splitPoints.empty() || splitPoints.back() != splitPoint) {
            splitPoints.push_back(splitPoint);
        }

        if (!split.isPoint()) {
            splitPolylines.push_back(cache(split));
            splitPoints.push_back(cache(split.end()));
        }

        if ((i + 1 < splits.size()) && (split.end() != splits[i + 1].start())) { // has next
            const PointsVector points = getSubPoints(
                edge.geom(), split.end(), splits[i + 1].start()
            );
            splitPolylines.push_back(
                std::make_shared<SplitPolyline>(Polyline2{points})
            );
        }
    }

    return std::make_unique<SplitEdgeEventData>(
            SourceEdgeID{edge.id(), true},
            /* isEditedEgde = */ false,
            edge.geom(),
            splitPoints,
            splitPolylines
    );
}

SplitPointPtr SplitEdgeEventMaker::cache(const SnapPoint& snapPoint)
{
    SplitPointPtr& splitPoint = points_[snapPoint];
    if (!splitPoint) {
        splitPoint = std::make_shared<SplitPoint>(snapPoint.geom(), snapPoint.node());
    }
    return splitPoint;
}

SplitPolylinePtr SplitEdgeEventMaker::cache(const SnapPolyline& snapPolyline)
{
    SplitPolylinePtr& splitPolyline = polylines_[snapPolyline];
    if (splitPolyline) {
        splitPolyline->isShared = true;
    } else {
        splitPolyline = std::make_shared<SplitPolyline>(snapPolyline.geom());
    }
    return splitPolyline;
}


} // namespace maps::wiki::topo::geom
} // namespace maps::wiki::topo
} // namespace maps::wiki
} // namespace maps
