#include "validation.h"

#include "geom_io.h"
#include "util.h"
#include "../util/io.h"

#include <yandex/maps/wiki/topo/exception.h>

#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/closest_point.h>
#include <maps/libs/geolib/include/intersection.h>
#include <maps/libs/geolib/include/projection.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>
#include <boost/optional/optional_io.hpp>
#include <deque>


namespace maps {
namespace wiki {
namespace topo {
namespace geom {


using namespace geolib3;


void checkSegmentsLimits(const Polyline2& polyline, const Limits<double>& limits)
{
    for (size_t i = 0; i < polyline.segmentsNumber(); ++i) {
        const Segment2& segment = polyline.segmentAt(i);
        const double segmentLength = length(segment);

        if (segmentLength < limits.min) {
            throw GeomProcessingErrorWithLocation(
                ErrorCode::DegenerateSegment, center(segment)
            ) << i << "-th segment of polyline "
              << "with length " << segmentLength << " is too short, "
              << "min possible value is " << limits.min;
        }

        if (limits.max && segmentLength > *limits.max) {
            throw GeomProcessingErrorWithLocation(
                ErrorCode::TooLongSegment, center(segment)
            ) << i << "-th segment of polyline "
              << "with length " << segmentLength << " is too long, "
              << "max possible value is " << limits.max;
        }
    }
}

void checkPolylineLimits(const Polyline2& polyline, const Limits<double>& limits)
{
    const double polylineLength = length(polyline);
    if (polylineLength < limits.min) {
        throw GeomProcessingErrorWithLocation(
            ErrorCode::DegenerateEdge,
            center(polyline)
            ) << "polyline with length " << polylineLength << " is too short, "
              << "min possible value is " << limits.max;
    }

    if (limits.max && polylineLength > *limits.max) {
        throw GeomProcessingErrorWithLocation(
            ErrorCode::TooLongEdge,
            center(polyline)
        ) << "polyline with length " << polylineLength << " is too long, "
          << "max possible value is " << limits.max;
    }
}


void checkMaxIntersectedEdges(const Intersector::Events& events, double limit)
{
    const size_t intersectedElemnts = events.intersectedEdgesEvents.size();
    if (intersectedElemnts > limit) {
        throw GeomProcessingError(
            ErrorCode::TooManyIntersectionsWithElement
        ) << "Too many edges are intersected "
          << "(" << intersectedElemnts << "), "
          << "but only " << limit << " possible";
    }
}

void checkMaxIntersectionsWithEdge(const Intersector::Events& events, double limit)
{
    for (auto& pair: events.intersectedEdgesEvents) {
        const SplitEdgeEventData& event = *pair.second;
        const size_t intersectionsNumber = event.splitPolylines.size() - 1;
        if (intersectionsNumber > limit) {
            throw GeomProcessingError(
                ErrorCode::TooManyIntersectionsWithNetwork
            )  << "Edge with id " << util::print(event.sourceId)
               << " is splitted into too many parts "
               << "(" << intersectionsNumber << "),"
               << " but only " << limit << " possible";
        }
    }
}

void checkUnusedNodes(const std::vector<Node>& nodes, DeleteNodeChecker check)
{
    for (const auto& node: nodes) {
        if (!check(node.id())) {
            throw GeomProcessingErrorWithLocation(
                ErrorCode::NodeDeletionForbidden,
                node.pos()
            )  << "Node with id " << node.id() << " is forbidden to delete";
        }
    }
}


namespace {

struct SegmentToIndex {
    std::deque<Segment2> segments;
    StaticGeometrySearcher<Segment2, size_t> index;
};

typedef std::shared_ptr<SegmentToIndex> SegmentToIndexPtr;

SegmentToIndexPtr buildSegmentToIndex(const Polyline2& polyline)
{
    SegmentToIndexPtr index = std::make_shared<SegmentToIndex>();
    for (size_t i = 0; i < polyline.segmentsNumber(); ++i) {
        index->segments.push_back(polyline.segmentAt(i));
        index->index.insert(&index->segments.back(), i);
    }
    index->index.build();
    return index;
}

typedef boost::optional<Point2> OptionalPoint;

OptionalPoint findPointOfOverlap(const Segment2& one, const Segment2& two, double tolerance)
{
    OptionalPoint point;

    auto check = [&point, &tolerance](const Segment2& s, const Point2& p)
    {
        const double factor = projectionFactor(s, p);
        if (near(s, p, tolerance) && (factor > EPS) && (factor < (1 - EPS))) {
            point = p;
        }
    };

    check(one, two.start());
    check(one, two.end());
    check(two, one.start());
    check(two, one.end());

    return point;
}

void checkNoSelfIntersection(const Polyline2& polyline, double tolerance,
    const SegmentToIndexPtr index)
{
    for (size_t i = 0; i < polyline.segmentsNumber(); ++i) {
        const Segment2& one = polyline.segmentAt(i);
        auto result = index->index.find(
            resizeByValue(one.boundingBox(), tolerance)
        );
        for (auto it = result.first; it != result.second; ++it) {
            const Segment2& two = it->geometry();
            const size_t j = it->value();
            if (j <= i + 1) {
                continue;
            }
            const auto intersectionPoints = intersection(one, two);
            if (intersectionPoints.size() == 2) {
                throw GeomProcessingErrorWithLocation(
                        ErrorCode::SelfOverlap,
                        center(
                            TwoPoints{intersectionPoints.front(), intersectionPoints.back()}
                        )
                    )  << "Self-ovrelap of segments " << i << " and " << j;

            } else if (intersectionPoints.size() == 1) {
                const Point2& point = intersectionPoints.front();
                const PointsVector& points = polyline.points();
                bool intersectionAtEndPoints =
                    (i == 0) && ((j + 1) == polyline.segmentsNumber()) &&
                    near(points.front(), point) &&
                    near(points.back(), point);

                if (!intersectionAtEndPoints) {
                    throw GeomProcessingErrorWithLocation(
                        ErrorCode::SelfIntersection,
                        point
                    )  << "Self-intersection of segments " << i << " and " << j;
                }
            }
        }
    }
}

void checkNoSelfOverlap(const Polyline2& polyline, double tolerance,
    const SegmentToIndexPtr index)
{
    for (size_t i = 0; i < polyline.segmentsNumber(); ++i) {
        const Segment2& one = polyline.segmentAt(i);
        auto result = index->index.find(
            resizeByValue(one.boundingBox(), tolerance)
        );
        for (auto it = result.first; it != result.second; ++it) {
            const Segment2& two = it->geometry();
            const size_t j = it->value();
            if (i >= j) {
                continue;
            }
            const OptionalPoint point = findPointOfOverlap(one, two, tolerance);
            if (point) {
                throw GeomProcessingErrorWithLocation(
                    ErrorCode::SelfOverlap,
                    *point
                )  << "Self-overlap of segments " << i << " and " << j;
            }
        }
    }
}

void checkNoSelfSnapping(const Polyline2& polyline, double tolerance,
    const SegmentToIndexPtr index)
{
    const PointsVector& points = polyline.points();
    for (size_t p = 0; p < points.size(); ++p) {
        const Point2& point = points[p];
        auto result = index->index.find(
            resizeByValue(point.boundingBox(), tolerance)
        );
        for (auto it = result.first; it != result.second; ++it) {
            const size_t s = it->value();
            if ((s <= p && p <= (1 + s)) ||
                (p == 0 && near(points.back(), point, tolerance)) ||
                ((p + 1) == points.size() && near(points.front(), point, tolerance)))
            {
                continue;
            }

            if (near(point, it->geometry(), tolerance)) {
                throw GeomProcessingErrorWithLocation(
                    ErrorCode::SelfOverlap,
                    point
                ) << "Vertex " << p << " too close to segment " << s;
            }
        }
    }
}

struct SegmentToEdgeId{
    std::deque<Segment2> segments;
    StaticGeometrySearcher<Segment2, EdgeID> index;
};

typedef std::shared_ptr<SegmentToEdgeId> SegmentToEdgeIdPtr;

typedef StaticGeometrySearcher<Point2, bool> NodeIndex;
typedef std::shared_ptr<NodeIndex> NodeIndexPtr;


SegmentToEdgeIdPtr buildSegmentToEdgeId(const Graph& graph, const Intersector::Events& events,
    double tolerance)
{
    SegmentToEdgeIdPtr index = std::make_shared<SegmentToEdgeId>();

    auto insertIntoIndex = [&](const Edge& edge)
    {
        auto it = events.intersectedEdgesEvents.find(edge.id());
        if (it != events.intersectedEdgesEvents.end()) {
            const SplitPolylinePtrVector& polylines = it->second->splitPolylines;
            for (const auto& polyline: polylines) {
                for (size_t i = 0; i < polyline->geom.segmentsNumber(); ++i) {
                    index->segments.push_back(polyline->geom.segmentAt(i));
                    index->index.insert(&index->segments.back(), edge.id());
                }
            }
        } else {
            for (size_t i = 0; i < edge.geom().segmentsNumber(); ++i) {
                index->segments.push_back(edge.geom().segmentAt(i));
                index->index.insert(&index->segments.back(), edge.id());
            }
        }
    };

    EdgeIDSet ignoredIds;
    if (events.editedEdgeEvent->sourceId.exists()) {
        ignoredIds.insert(events.editedEdgeEvent->sourceId.id());
    }
    for (const auto& splitPolyline: events.editedEdgeEvent->splitPolylines) {
        const Polyline2& polyline = splitPolyline->geom;
        const BoundingBox polylineBox = resizeByValue(polyline.boundingBox(), tolerance);
        for (const auto& edge: graph.edges()) {
            if (!ignoredIds.count(edge.id()) &&
                intersects(polylineBox, resizeByValue(edge.geom().boundingBox(), tolerance)))
            {
                ignoredIds.insert(edge.id());
                insertIntoIndex(edge);
            }
        }
    }
    index->index.build();
    return index;
}

NodeIndexPtr buildNodeIndex(const SplitEdgeEventData& event)
{
    NodeIndexPtr index = std::make_shared<NodeIndex>();
    for (const auto& splitPoint: event.splitPoints) {
        index->insert(&splitPoint->geom, true);
    }
    for (const auto& splitPolyline: event.splitPolylines) {
        if (!splitPolyline->isShared) {
            continue;
        }
        const Polyline2& polyline = splitPolyline->geom;
        for (size_t i = 1; i < polyline.pointsNumber() - 1; ++i) {
            index->insert(&polyline.pointAt(i), true);
        }
    }
    index->build();
    return index;
}

} // end of anonymous namespace


void checkNoSelfInteraction(const Polyline2& polyline, double tolerance)
{
    const SegmentToIndexPtr index = buildSegmentToIndex(polyline);

    checkNoSelfIntersection(polyline, tolerance, index);
    checkNoSelfOverlap(polyline, tolerance, index);
    checkNoSelfSnapping(polyline, tolerance, index);
}


void checkNoInteraction(const Graph& graph, const Intersector::Events& events, double tolerance)
{
    const SegmentToEdgeIdPtr segmentIndex = buildSegmentToEdgeId(graph, events, tolerance);
    const NodeIndexPtr nodeIndex = buildNodeIndex(*events.editedEdgeEvent);

    for (const auto& splitPolyline: events.editedEdgeEvent->splitPolylines) {
        const Polyline2& polyline = splitPolyline->geom;
        for (size_t i = 0; i < polyline.segmentsNumber(); ++i) {
            const Segment2& segment = polyline.segmentAt(i);
            const auto segmentResult = segmentIndex->index.find(
                resizeByValue(segment.boundingBox(), EPS)
            );
            for (auto it = segmentResult.first; it != segmentResult.second; ++it) {
                const auto& edgeId = it->value();
                const auto& edgeSegment = it->geometry();
                const auto points = intersection(segment, edgeSegment);
                if (points.size() == 1) {
                    const Point2& point = points.front();
                    const auto nodeResult = nodeIndex->find(
                        resizeByValue(point.boundingBox(), EPS)
                    );
                    if (nodeResult.first == nodeResult.second) {
                        throw GeomProcessingErrorWithLocation(
                            ErrorCode::UnexpectedIntersection,
                            point
                        ) << " edited edge intersects edge " << edgeId;
                    }
                } else if (points.size() == 2) {
                    if (!splitPolyline->isShared ||
                            !near(segment.start(), points.front()) ||
                            !near(segment.end(), points.back()))
                    {
                        throw GeomProcessingErrorWithLocation(
                            ErrorCode::UnexpectedIntersection,
                            center(TwoPoints{points.front(), points.back()})
                        ) << " edited edge intersects edge " << edgeId
                          << " with overlapping";
                    }
                }
            }
        }
    }
}

} // namespace geom
} // namespace topo
} // namespace wiki
} // namespace maps
