#pragma once

#include "index/spatial_index.h"

#include <yandex/maps/wiki/topo/cache.h>
#include <yandex/maps/wiki/topo/node.h>
#include <yandex/maps/wiki/topo/edge.h>

#include <boost/optional.hpp>

#include <unordered_map>
#include <memory>
#include <type_traits>

namespace maps {
namespace wiki {
namespace topo {

template <class ID, class Object>
class ObjectPtrMap {
public:
    typedef std::unordered_map<ID, std::unique_ptr<Object>> Type;
};

typedef ObjectPtrMap<NodeID, Node>::Type NodePtrMap;
typedef ObjectPtrMap<EdgeID, Edge>::Type EdgePtrMap;

// concrete iterators implementations

template <class ID, class Object>
class IteratorImpl {
public:
    typedef typename ObjectPtrMap<
        ID,
        typename std::remove_const<Object>::type
    >::Type::const_iterator
    BaseIterator;

    typedef Object& Reference;

    IteratorImpl(BaseIterator it)
        : baseIt(it)
    {}

    BaseIterator baseIt;
};

typedef IteratorImpl<NodeID, const Node> NodeConstIteratorImpl;
typedef IteratorImpl<EdgeID, const Edge> EdgeConstIteratorImpl;

class Graph {
public:
    Graph(std::unique_ptr<index::SpatialIndex> index, double tolerance);

    const Node& node(NodeID nodeId) const;
    Node& node(NodeID nodeId);
    bool nodeExists(NodeID nodeId) const;

    const Edge& edge(EdgeID edgeId) const;
    Edge& edge(EdgeID edgeId);
    bool edgeExists(EdgeID edgeId) const;

    // добавить объекты, которые вернул Storage
    // одновременно идет добавление в индекс
    void addNode(const Node& node);
    void addEdge(const Edge& edge);
    // с одновременным удалением из индекса
    void deleteNode(NodeID node);
    void deleteEdge(EdgeID edge);

    OptionalNodeID commonNode(EdgeID edge1, EdgeID edge2) const;

    void setEdgeGeom(EdgeID edgeId, const geolib3::Polyline2& newGeom);
    void moveNode(NodeID nodeId, const geolib3::Point2& newPos);

    EdgePtrVector incidentEdges(Node& node);
    EdgePtrVector incidentEdges(NodeID nodeId);

    NodeConstIteratorRange nodes() const;
    EdgeConstIteratorRange edges() const;

    const index::SpatialIndex& index() const;

private:
    void checkNode(NodeID nodeId) const;
    void checkNodes(const NodeIDSet& nodeIds) const;
    void checkEdge(EdgeID edgeId) const;
    void checkEdges(const EdgeIDSet& edgeIds) const;

    NodePtrMap nodes_;
    EdgePtrMap edges_;

    std::unique_ptr<index::SpatialIndex> index_;
    const double tolerance_;
};

} // namespace maps::wiki::topo
} // namespace maps::wiki
} // namespace maps
