#pragma once

#include <yandex/maps/wiki/topo/common.h>
#include <yandex/maps/wiki/topo/edge.h>
#include <yandex/maps/wiki/topo/node.h>

#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/bounding_box.h>

#include <boost/iterator/iterator_facade.hpp>

#include <memory>

namespace maps {
namespace wiki {
namespace topo {

template <class ID, class Object> class IteratorImpl;

template <class ID, class Object>
class Iterator : public boost::iterator_facade<
        Iterator<ID, Object>,
        Object, // value
        boost::forward_traversal_tag, // categoryOrTraversal
        Object&, // ref
        std::ptrdiff_t> // difference_type
{
public:
    Iterator(std::unique_ptr<IteratorImpl<ID, Object> > impl);
    Iterator(const Iterator<ID, Object>& other);
    Iterator(Iterator<ID, Object>&& other);
    Iterator& operator=(const Iterator<ID, Object>& other);
    Iterator& operator=(Iterator<ID, Object>&& other);
    ~Iterator();

private:
    friend class boost::iterator_core_access;

    Object& dereference() const;
    bool equal(const Iterator<ID, Object>& other) const;
    void increment();

    std::unique_ptr<IteratorImpl<ID, Object> > impl_;
};

typedef Iterator<NodeID, const Node> NodeConstIterator;
typedef Iterator<EdgeID, const Edge> EdgeConstIterator;

template <class ID, class Object>
class Range
{
public:
    Range(const Iterator<ID, Object>& begin, const Iterator<ID, Object>& end)
        : begin_(begin)
        , end_(end)
    {}

    Iterator<ID, Object> begin() const { return begin_; }
    Iterator<ID, Object> end() const { return end_; }

private:
    Iterator<ID, Object> begin_;
    Iterator<ID, Object> end_;
};

typedef Range<NodeID, const Node> NodeConstIteratorRange;
typedef Range<EdgeID, const Edge> EdgeConstIteratorRange;

class CacheImpl;

class Storage;
class Editor;
class FaceValidator;

class Cache {
public:
    Cache(Storage& storage, double tolerance);

    ~Cache();

    const Node& node(NodeID id) const;
    bool nodeExists(NodeID id) const;

    const Edge& edge(EdgeID id) const;
    bool edgeExists(EdgeID id) const;

    NodeConstIteratorRange nodes() const;
    EdgeConstIteratorRange edges() const;

    IncidencesByNodeMap incidencesByNodes(const NodeIDSet& nodeIds);
    IncidencesByEdgeMap incidencesByEdges(const EdgeIDSet& edgeIds);

    /// Compute circuit from given edges.
    boost::optional<Circuit> circuit(const EdgeIDSet& edgeIds, bool withNodes);

    /// Load data

    void loadObjects(const geolib3::BoundingBox& bbox);
    void loadObjects(const geolib3::Point2& center,
        double width, double height);
    void loadObjects(const geolib3::Polyline2& polyline,
        double withinDistance);

    void loadByNodes(const NodeIDSet& nodeIds);
    void loadByEdges(const EdgeIDSet& edgeIds);

    void loadNode(NodeID nodeId);
    void loadEdge(EdgeID edgeId);

    std::unique_ptr<Editor> editor();
    std::unique_ptr<FaceValidator> faceValidator();

private:
    std::unique_ptr<CacheImpl> impl_;
};

} // namespace topo
} // namespace wiki
} // namespace maps
