#pragma once

#include <yandex/maps/wiki/topo/common.h>
#include <yandex/maps/wiki/topo/events.h>

#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polyline.h>

#include <boost/noncopyable.hpp>

#include <functional>
#include <map>
#include <memory>
#include <vector>

namespace maps {
namespace wiki {
namespace topo {

template <class Event>
class TopoCallback {
public:
    virtual ~TopoCallback() {}
    virtual void process(const Event& /*event*/) const {}
};

template <class Request, class Event>
class Callback {
public:
    virtual ~Callback() {}
    virtual void processRequest(Request&) const {}
    virtual void processEvent(const Event&) const {}
};

/// Topology level callbacks

typedef TopoCallback<AddEdgeEvent> TopoAddEdgeCallback;
typedef TopoCallback<MoveEdgeEvent> TopoMoveEdgeCallback;
typedef TopoCallback<DeleteEdgeEvent> TopoDeleteEdgeCallback;

/// Domain level callbacks

typedef Callback<SplitEdgeRequest, SplitEdgeEvent> SplitEdgeCallback;
typedef Callback<MergeEdgesRequest, MergeEdgesEvent> MergeEdgesCallback;
typedef Callback<DeleteEdgeRequest, DeleteEdgeEvent> DeleteEdgeCallback;
typedef Callback<MergeNodesRequest, MergeNodesEvent> MergeNodesCallback;

class EditorImpl;

class Editor : public boost::noncopyable {
public:
    explicit Editor(std::unique_ptr<EditorImpl> impl);

    ~Editor();

    static double maxInteractionDistance(
        double gravityDistance);

    static double maxInteractionDistance(
        const TopologyRestrictions& restrictions);

    struct NodeData {
        NodeID id;
        geolib3::Point2 pos;
    };

    struct EdgeData {
        SourceEdgeID id;
        geolib3::PointsVector splitPoints;
        geolib3::Polyline2 geom;
    };

    struct TopologyData {
        std::vector<NodeData> nodesData;
        std::vector<EdgeData> edgesData;
    };

    void saveEdge(
        const EdgeData& data,
        const TopologyRestrictions& restrictions);

    void saveNode(
        const NodeData& data,
        const TopologyRestrictions& restrictions);

    void saveObjects(
        const TopologyData& data,
        const TopologyRestrictions& restrictions);

    typedef std::map<EdgeID, EdgeIDSet> SplitEdgesMap;

    struct CreateIntersectionsResult
    {
        SplitEdgesMap splitEdgeIds;
        geolib3::PolylinesVector adjustedGeoms;
    };

    CreateIntersectionsResult
    createIntersections(
        const geolib3::PolylinesVector& geoms,
        const TopologyRestrictions& restrictions);


    void deleteEdge(EdgeID id);

    void mergeEdges(NodeID commonNodeId);


    void snapNodes(
        const NodeIDSet& nodeIds,
        const TopologyRestrictions& restrictions);


    void registerTopoCallback(std::unique_ptr<TopoAddEdgeCallback> callback);
    void registerTopoCallback(std::unique_ptr<TopoMoveEdgeCallback> callback);
    void registerTopoCallback(std::unique_ptr<TopoDeleteEdgeCallback> callback);

    void registerCallback(std::unique_ptr<SplitEdgeCallback> callback);
    void registerCallback(std::unique_ptr<DeleteEdgeCallback> callback);
    void registerCallback(std::unique_ptr<MergeEdgesCallback> callback);
    void registerCallback(std::unique_ptr<MergeNodesCallback> callback);

private:
    std::unique_ptr<EditorImpl> impl_;
};

} // namespace topo
} // namespace wiki
} // namespace maps
