#pragma once

#include <yandex/maps/wiki/topo/common.h>

#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/point.h>

#include <boost/optional.hpp>

#include <memory>


namespace maps {
namespace wiki {
namespace topo {


class Edge;
class Node;

/**
 * Topology changes events.
 */

/**
 * New edge addition event.
 * Edge is represented by its geometry (polyline) and start and end nodes.
 * New edge is added only to already created nodes, and its geometry
 *   has no intersections with existing edges at internal points.
 * Topology-level event.
 */

struct AddEdgeEventData;

class AddEdgeEvent
{
public:
    explicit AddEdgeEvent(const AddEdgeEventData& data);

    EdgeID id() const;
    EdgeID sourceId() const;
    bool sourceExists() const;
    NodeID startNode() const;
    NodeID endNode() const;
    const geolib3::Polyline2& geom() const;

private:
    const AddEdgeEventData& data_;
};

/**
 * Delete edge event.
 * Changes connectivity of its incident nodes.
 * Can be used both as topology- and domain-level events.
 */

struct DeleteEdgeEventData;

class DeleteEdgeRequest
{
public:
    explicit DeleteEdgeRequest(DeleteEdgeEventData& data);

    EdgeID id() const;
    NodeID startNodeId() const;
    NodeID endNodeId() const;

protected:
    DeleteEdgeEventData& data_;
};

class DeleteEdgeEvent
{
public:
    explicit DeleteEdgeEvent(const DeleteEdgeEventData& data);

    EdgeID id() const;
    NodeID startNodeId() const;
    NodeID endNodeId() const;

protected:
    const DeleteEdgeEventData& data_;
};


/**
 * Edge are splitted in N parts (N can be equal 1 in the full overlapping case).
 * Contains N+1 shared splitting points and N shared splitting polylines.
 * Two edges can share one polyline splitting.
 *
 * It is guranteed that no changes between split request and split event processing callbacks.
 * It is guranteed that split before processing split event callback
 * all objects are created and ids are set.
 * Splitting polylines of one edge can have different direction.
 */

/**
 * Struct representing edge splitting point.
 * Before events processing nodeId is not set
 * if split point doesn't snap to any of existing nodes.
 * In processing ids is set as new nodes are created.
 */
struct SplitPoint
{
    explicit SplitPoint(const geolib3::Point2& geom, OptionalNodeID nodeId = boost::none)
        : geom(geom)
        , nodeId(nodeId)
    {}

    geolib3::Point2 geom;
    OptionalNodeID nodeId;
};

typedef std::shared_ptr<SplitPoint> SplitPointPtr;
typedef std::vector<SplitPointPtr> SplitPointPtrVector;


/**
 * Struct representing edge splitting part,
 * Before events processing edgeId is not set
 * In processing ids is set as new edges are created
 * or old edges are moved
 * shared is true if part is common for two edges.
 *
 * Domain-level event.
 */

struct SplitPolyline
{
    explicit SplitPolyline(const geolib3::Polyline2& geom,
        OptionalEdgeID edgeId = boost::none, bool isShared = false)
        : geom(geom)
        , edgeId(edgeId)
        , isShared(isShared)
    {}

    geolib3::Polyline2 geom;
    OptionalEdgeID edgeId;
    bool isShared;
};


typedef std::shared_ptr<SplitPolyline> SplitPolylinePtr;
typedef std::vector<SplitPolylinePtr> SplitPolylinePtrVector;


struct SplitEdgeEventData;

class SplitEdgeRequest
{
public:
    explicit SplitEdgeRequest(SplitEdgeEventData& data);

    EdgeID sourceId() const;
    bool sourceExists() const;
    bool isEditedEdge() const;

    const SplitPointPtrVector& splitPoints() const;
    const SplitPolylinePtrVector& splitPolylines() const;
    const OptionalIndex& partToKeepID() const;
    void selectPartToKeepID(size_t partNum);

protected:
    SplitEdgeEventData& data_;
};

class SplitEdgeEvent
{
public:
    explicit SplitEdgeEvent(const SplitEdgeEventData& data);

    EdgeID sourceId() const;
    bool sourceExists() const;

    std::vector<EdgeID> edgeIds() const;
    std::vector<NodeID> nodeIds() const;

    const OptionalIndex& partToKeepID() const;

protected:
    const SplitEdgeEventData& data_;
};

/**
 * Merge two edges sharing a node which has no other incident edges.
 * Domain-level event.
 */

struct MergeEdgesEventData;

class MergeEdgesRequest
{
public:
    explicit MergeEdgesRequest(MergeEdgesEventData& data);

    /// Id of edge to keep as result of merge
    EdgeID mergedId() const;
    /// Id of edge to be deleetd
    EdgeID deletedId() const;
    /// Common node id of merged edges
    NodeID nodeId() const;
    geolib3::Polyline2 newGeom() const;

    /// Swap merged and deleted edge ids.
    void swapMergedAndDeleted();

protected:
    MergeEdgesEventData& data_;
};

class MergeEdgesEvent
{
public:
    explicit MergeEdgesEvent(const MergeEdgesEventData& data);

    EdgeID mergedId() const;
    EdgeID deletedId() const;
    NodeID nodeId() const;

protected:
    const MergeEdgesEventData& data_;
};


/**
 * Event is created when existing edge geometry modification leads to
 *   change of its start and end nodes.
 * NO newly created intersection are reported. These intersections
 *   can be determined by SplitEdgeEvent following this event.
 * Topology-level event.
 */

struct MoveEdgeEventData;

class MoveEdgeEvent
{
public:
    explicit MoveEdgeEvent(const MoveEdgeEventData& data);

    EdgeID id() const;
    /// Set to edge's new start node if it is changed
    OptionalNodeID newStartNode() const;
    /// Set to edge's new end node if it is changed
    OptionalNodeID newEndNode() const;
    /// New edge geom
    const geolib3::Polyline2& newGeom() const;

    /// Edge's start node before move.
    NodeID oldStartNode() const;
    /// Edge's end node before move.
    NodeID oldEndNode() const;

protected:
    const MoveEdgeEventData& data_;
};

/**
 * Merge two nodes that became within each other's gravity.
 * mergedId() node will be left, deletedId() node will be deleted.
 * All edges from deletedId() node will be moved to mergedId() node and their
 *   geometries will be adjusted causing one MoveEdge event per incident edge
 *   between request and event.
 * Affected edges geometries will change after request prior to event.
 * Domain-level event.
 */

struct MergeNodesEventData;

class MergeNodesRequest
{
public:
    explicit MergeNodesRequest(MergeNodesEventData& data);

    NodeID mergedId() const;
    NodeID deletedId() const;
    const geolib3::Point2& pos() const;

    /// Select other node to be kept as merged.
    void swapMergedAndDeleted();

protected:
    MergeNodesEventData& data_;
};

class MergeNodesEvent
{
public:
    explicit MergeNodesEvent(const MergeNodesEventData& data);

    NodeID mergedId() const;
    NodeID deletedId() const;

protected:
    const MergeNodesEventData& data_;
};

} // namespace topo
} // namespace wiki
} // namespace maps
