#pragma once

#include <yandex/maps/wiki/topo/common.h>
#include <yandex/maps/wiki/topo/node.h>
#include <yandex/maps/wiki/topo/edge.h>

#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polyline.h>

#include <boost/noncopyable.hpp>

#include <memory>
#include <vector>
#include <unordered_map>

namespace maps {
namespace wiki {
namespace topo {

/// Must NOT return objects that were already deleted

class Storage : public boost::noncopyable {
public:
    virtual ~Storage() {}

    virtual NodeIDSet nodeIds(const geolib3::BoundingBox& bbox) = 0;
    virtual EdgeIDSet edgeIds(const geolib3::BoundingBox& bbox) = 0;

    /// nodeIds must contain only existent node ids
    virtual IncidencesByNodeMap incidencesByNodes(const NodeIDSet& nodeIds) = 0;
    /// edgeIds must contain only existent edge ids
    virtual IncidencesByEdgeMap incidencesByEdges(const EdgeIDSet& edgeIds) = 0;

    /// incidences as to initial storage state, before any modifications
    /// edges deleted during storage lifetime are allowed to appear in edgeIds
    virtual IncidencesByEdgeMap originalIncidencesByEdges(const EdgeIDSet& edgeIds) = 0;

    virtual EdgeIDSet incidentEdges(NodeID endNode) = 0;
    virtual IncidentNodes incidentNodes(EdgeID edgeId) = 0;

    virtual Node node(NodeID id) = 0;
    virtual Edge edge(EdgeID id) = 0;

    virtual NodeVector nodes(const NodeIDSet& nodeIds) = 0;
    virtual EdgeVector edges(const EdgeIDSet& edgeIds) = 0;

    virtual NodeID newNodeId() = 0;
    virtual EdgeID newEdgeId() = 0;

    virtual Node createNode(const geolib3::Point2& pos) = 0;
    virtual void updateNodePos(NodeID id, const geolib3::Point2& pos) = 0;
    virtual void deleteNode(NodeID id) = 0;

    virtual Edge createEdge(
        const geolib3::Polyline2& geom,
        const IncidentNodes& nodes) = 0;
    virtual void updateEdgeGeom(EdgeID id,
        const geolib3::Polyline2& geom,
        const topo::IncidentNodes& nodes) = 0;
    virtual void deleteEdge(EdgeID id) = 0;

    virtual bool nodeCanBeDeleted(NodeID id) = 0;
    virtual bool edgeCanBeDeleted(EdgeID id) = 0;

    // face related methods

    // load related faces by given edge ids
    virtual FacesByEdgeMap facesByEdges(const EdgeIDSet& edgeIds) = 0;

    // get added, changed and removed edges of a face
    virtual FaceDiff faceDiff(FaceID faceId) = 0;

    // get face edges if available
    virtual boost::optional<EdgeIDSet> tryGetFaceEdges(FaceID faceId) = 0;
    // load face edges
    virtual EdgeIDSet getFaceEdges(FaceID faceId) = 0;

protected:
    Storage() {}
};

} // namespace topo
} // namespace wiki
} // namespace maps
