#include <yandex/maps/wiki/topo/node.h>

#include "node_impl.h"

#include <yandex/maps/wiki/topo/edge.h>
#include <yandex/maps/wiki/topo/exception.h>

#include <map>

namespace maps {
namespace wiki {
namespace topo {

Node::Node(NodeID id, const geolib3::Point2& pos)
    : impl_(new Impl(id, pos))
{}

Node::Node(const Node& oth)
    : impl_(new Impl(*oth.impl_))
{}

Node& Node::operator=(const Node& oth)
{
    impl_.reset(new Impl(*oth.impl_));
    return *this;
}

Node::Node(Node&& oth)
    : impl_(std::move(oth.impl_))
{}

Node& Node::operator=(Node&& oth)
{
    impl_ = std::move(oth.impl_);
    return *this;
}

Node::~Node()
{ /* impl_ */ }

NodeID Node::id() const
{
    return impl_->id();
}

EdgeConstPtrVector Node::incidentEdges() const
{
    EdgePtrVector incidentEdges = impl_->incidentEdges();
    return EdgeConstPtrVector(incidentEdges.begin(), incidentEdges.end());
}

EdgeIDVector Node::incidentEdgeIds() const
{
    EdgeIDVector ids;
    ids.reserve(impl_->incidentEdges().size());
    for (const auto& edge: impl_->incidentEdges()) {
        ids.push_back(edge->id());
    }
    return ids;
}


const geolib3::Point2& Node::pos() const
{
    return impl_->pos();
}

size_t Node::degree() const
{
    return impl_->startIncidentEdges().size() + impl_->endIncidentEdges().size();
}

bool Node::isIsolated() const
{
    return degree() == 0;
}

bool Node::isPendant() const
{
    return degree() == 1;
}

bool Node::isIncident(EdgeID edge) const
{
    return isStartOf(edge) || isEndOf(edge);
}

bool Node::isStartOf(EdgeID edgeId) const
{
    return impl_->isStartOf(edgeId);
}

bool Node::isEndOf(EdgeID edgeId) const
{
    return impl_->isEndOf(edgeId);
}

/// Node::Impl

Node::Impl::Impl(NodeID id, const geolib3::Point2& pos)
    : id_(id)
    , pos_(pos)
    , areIncidentEdgesLoaded_(false)
{
    REQUIRE(id_, "Empty node id");
}

EdgePtrVector Node::Impl::incidentEdges()
{
    std::map<EdgeID, Edge*> edgesMap;
    for (auto edgePtr : startEdges_) {
        edgesMap.insert({edgePtr->id(), edgePtr});
    }
    for (auto edgePtr : endEdges_) {
        edgesMap.insert({edgePtr->id(), edgePtr});
    }
    EdgePtrVector edges;
    edges.reserve(edgesMap.size());
    for (const auto& edgePtrPair : edgesMap) {
        edges.push_back(edgePtrPair.second);
    }
    return edges;
}

bool Node::Impl::isStartOf(EdgeID edgeId) const
{
    for (const auto& edgePtr : startEdges_) {
        if (edgePtr->id() == edgeId) {
            return true;
        }
    }
    return false;
}

bool Node::Impl::isEndOf(EdgeID edgeId) const
{
    for (const auto& edgePtr : endEdges_) {
        if (edgePtr->id() == edgeId) {
            return true;
        }
    }
    return false;
}

void Node::Impl::addEdge(Edge* edge, IncidenceType incidenceType)
{
    REQUIRE(edge->startNode() == id_ || edge->endNode() == id_,
        " edge with id " << edge->id() << " is not incident to node with id " << id_);

    const bool isStart = incidenceType == IncidenceType::Start;
    REQUIRE(
        !((isStart && isStartOf(edge->id())) || (!isStart && isEndOf(edge->id()))),
        " edge with id " << edge->id() << " is already incident to node with id " << id_);
    (isStart ? startEdges_ : endEdges_).push_back(edge);
}

void Node::Impl::removeEdge(EdgeID edgeId, IncidenceType incidenceType)
{
    EdgePtrVector& edges =
        incidenceType == IncidenceType::Start ? startEdges_ : endEdges_;
    for (EdgePtrVector::iterator edgeIt = edges.begin();
        edgeIt != edges.end();
        ++edgeIt)
    {
        if ((*edgeIt)->id() == edgeId) {
            edges.erase(edgeIt);
            return;
        }
    }
    throw LogicError() << " edge " << edgeId << " is not incident to node " << id_;
}

} // namespace topo
} // namespace wiki
} // namespace maps
