#include <yandex/maps/wiki/unittest/query_helpers_common.h>

#include "util.h"

#include <iomanip>

namespace maps::wiki::unittest {

namespace {

std::vector<size_t> columnWidths(const pqxx::result& rows)
{
    std::vector<size_t> result(rows.columns(), 0);

    for (size_t columnIx = 0; columnIx < rows.columns(); ++columnIx) {
        result[columnIx] = std::strlen(rows.column_name(columnIx));
    }

    for (const auto& row: rows) {
        for (auto columnIx = 0u; columnIx < rows.columns(); ++columnIx) {
            result[columnIx] = std::max(result[columnIx], std::strlen(row[columnIx].c_str()));
        }
    }

    return result;
}

} // namespace

void printQueryResult(std::ostream& os, const pqxx::result& rows)
{
    auto colWidths = columnWidths(rows);

    os << std::left << "|";
    for (size_t columnIx = 0; columnIx < rows.columns(); ++columnIx) {
        os << "| " << std::setw(colWidths[columnIx]) << rows.column_name(columnIx) << " ";
    }
    os << "||\n";

    os << std::right;
    for (const auto& row: rows) {
        os << "|";
        for (auto columnIx = 0u; columnIx < rows.columns(); ++columnIx) {
            os << "| " << std::setw(colWidths[columnIx]) << row[columnIx].c_str() << " ";
        }
        os << "||\n";
    }
}

std::pair<bool, std::string>
compareRows(
    const std::string& lhsQuery, const std::string& rhsQuery,
    const pqxx::result& lhsRows, const pqxx::result& rhsRows)
{
    auto fail =
        [&](const std::string& message) {
            std::ostringstream oss;
            oss << message << "\n";
            oss << "Left query:\n" << lhsQuery << "\nreturned:\n";
            printQueryResult(oss, lhsRows);
            oss << "Right query:\n" << rhsQuery << "\nreturned:\n";
            printQueryResult(oss, rhsRows);
            return std::make_pair(false, oss.str());
        };

    if (lhsRows.size() != rhsRows.size()) {
        return fail(
            "Different rows number: " +
            std::to_string(lhsRows.size()) + " != " + std::to_string(rhsRows.size()));
    }

    if (lhsRows.columns() != rhsRows.columns()) {
        return fail(
            "Different columns number: " +
            std::to_string(lhsRows.columns()) + " != " + std::to_string(rhsRows.columns()));
    }

    for (size_t rowIx = 0; rowIx < lhsRows.size(); ++rowIx) {
        for (auto columnIx = 0u; columnIx < lhsRows[rowIx].size(); ++columnIx) {
            if (lhsRows[rowIx][columnIx] != rhsRows[rowIx][columnIx]) {
                return fail(
                    "Different data at row " + std::to_string(rowIx) + " "
                    "column " + std::to_string(columnIx) + ":\n"
                    "'" + lhsRows[rowIx][columnIx].c_str() +
                    "' != '" +
                    rhsRows[rowIx][columnIx].c_str() + "'");
            }
        }
    }

    return {true, {}};
}

} // namespace maps::wiki::unittest
