#include <yandex/maps/wiki/validator/changed_objects.h>
#include <yandex/maps/wiki/common/revision_utils.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/commit.h>
#include <yandex/maps/wiki/revision/filters.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>

#include <maps/libs/log8/include/log8.h>

namespace maps {
namespace wiki {
namespace validator {

namespace rf = maps::wiki::revision::filters;

ObjectIdSet findChangedObjectsInReleaseBranch(
    pgpool3::Pool& pgPool,
    revision::DBID branchId,
    revision::DBID commitId)
{
    auto txn = common::getReadTransactionForCommit(
        pgPool,
        branchId,
        commitId,
        [](const std::string& s) { INFO() << s; });

    auto branch = revision::BranchManager(*txn).load(branchId);
    REQUIRE(branch.type() == revision::BranchType::Stable || branch.type() == revision::BranchType::Archive,
        "Can't find changed objects in the branch of '" << branch.type() << "' type");

    revision::RevisionsGateway gateway(*txn, branch);

    auto objectsFilter =
        rf::CommitAttr::stableBranchId() == branchId
        && !rf::CommitAttr::isTrunk()
        && rf::CommitAttr::id() <= commitId
        && rf::ObjRevAttr::isNotRelation()
        && rf::ObjRevAttr::isNotDeleted();

    std::set<validator::TId> commitIds;

    validator::ObjectIdSet objectIds;
    for (const auto revId : gateway.reader().loadRevisionIds(objectsFilter)) {
        objectIds.insert(revId.objectId());
        commitIds.insert(revId.commitId());
    }

    auto relationsFilter =
        rf::CommitAttr::stableBranchId() == branchId
        && !rf::CommitAttr::isTrunk()
        && rf::CommitAttr::id() <= commitId
        && rf::ObjRevAttr::isRelation();

    for (const auto& rev : gateway.reader().loadRevisions(relationsFilter)) {
        objectIds.insert(rev.data().relationData->masterObjectId());
        objectIds.insert(rev.data().relationData->slaveObjectId());
        commitIds.insert(rev.id().commitId());
    }

    INFO() << "Commits in the release branch " << commitIds.size();

    std::map<validator::TId, size_t> authorIdToCommitCount;;

    auto commits = revision::Commit::load(*txn, rf::CommitAttr::id().in(commitIds));
    for (const auto& commit : commits) {
        INFO() << "Commit " << commit.id()
            << " author " << commit.createdBy()
            << " attributes " << common::join(commit.attributes(),
                [](const auto& pair) {
                    return pair.first + "=" + pair.second;
                },
                ", ");
        authorIdToCommitCount[commit.createdBy()]++;
    }

    for (const auto& [uid, count] : authorIdToCommitCount) {
        INFO() << "User " << uid << " made " << count << " commits";
    }

    return objectIds;
}

} // namespace validator
} // namespace wiki
} // namespace maps
