#pragma once

#include <maps/wikimap/mapspro/libs/validator/common/utils.h>

#include <yandex/maps/wiki/threadutils/executor.h>
#include <yandex/maps/wiki/validator/common.h>
#include <yandex/maps/wiki/validator/check_context.h>

#include <boost/none.hpp>
#include <algorithm>
#include <iterator>

namespace maps {
namespace wiki {
namespace validator {

template<class Object>
void visitOne(
        std::function<void(const Object*)> visitor,
        const Object* object,
        CheckContext& context)
{
    try {
        visitor(object);
    } catch (...) {
        context.fatal(
            "visit-internal-error",
            boost::none,
            { object->id() });
        logException(log8::Level::WARNING,
                     "while visiting object " +
                     std::to_string(object->id()));
    }
}

template<class Coll, class RangeVisitor>
void batchVisitCollection(
        const Coll& collection,
        RangeVisitor visitor,
        size_t batchSize,
        ThreadPool& pool)
{
    Executor executor;
    auto batchBegin = std::begin(collection);
    auto size = collection.size();
    for (size_t i = 0; i < size; i += batchSize) {
        auto batchEnd = batchBegin;
        std::advance(
                batchEnd,
                std::min(batchSize, size - i));

        executor.addTask([=]{ visitor(batchBegin, batchEnd); });

        batchBegin = std::move(batchEnd);
    }

    executor.executeAll(pool);
}

template<class Object>
void batchVisitVec(
        const std::vector<const Object*>& objects,
        std::function<void(const Object*)> visitor,
        size_t batchSize,
        CheckContext& context,
        ThreadPool& pool)
{
    typedef decltype(objects.cbegin()) CObjIt;
    auto visitFunc = [&](CObjIt begin, CObjIt end)
    {
        for (auto it = begin; it != end; ++it) {
            visitOne<Object>(visitor, *it, context);
        }
    };
    batchVisitCollection(objects, visitFunc, batchSize, pool);
}

} // namespace validator
} // namespace wiki
} // namespace maps
