#pragma once

#include <yandex/maps/wiki/validator/common.h>
#include <yandex/maps/wiki/revision/snapshot.h>
#include <yandex/maps/coverage5/coverage.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/intersection.h>
#include <maps/libs/geolib/include/linear_ring.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/spatial_relation.h>
#include <maps/libs/pgpool/include/pgpool3.h>

#include <boost/optional.hpp>

#include <map>
#include <memory>
#include <vector>

#define YANDEX_MAPS_WIKI_VALIDATOR_AREA_OF_INTEREST_INL
#include "area_of_interest_inl.h"
#undef YANDEX_MAPS_WIKI_VALIDATOR_AREA_OF_INTEREST_INL

namespace maps {
namespace wiki {
namespace validator {

using DBID2PolygonVector = std::unordered_map<DBID, PolygonVector>;

class AreaOfInterest
{
public:
    AreaOfInterest();
    AreaOfInterest(AreaOfInterest&&) = default;
    AreaOfInterest& operator=(AreaOfInterest&&) = default;
    ~AreaOfInterest();

    bool empty() const { return includedPrimaryPolygonGroups_.empty(); }

    void setBuffer(double buffer);

    void setCoverageDir(std::string dir) { coverageDir_ = std::move(dir); }

    template<typename TPolygon>
    void addPolygon(TPolygon&& polygon)
    {
        includedPrimaryPolygonGroups_[DBID_USER_AOI].emplace_back(std::forward<TPolygon>(polygon));
    }

    void addByIds(
        pgpool3::Pool& pgPool,
        DBID branchId,
        DBID commitId,
        const std::vector<DBID>& oids);

    void build();

    const boost::optional<geolib3::BoundingBox>& boundingBox() const
    {
        return boundingBox_;
    }

    const std::vector<geolib3::BoundingBox>& boundingBoxesBuffered() const
    {
        return bufferedBoundingBoxes_;
    }

    template<typename TGeom>
    bool intersects(const TGeom& geom) const
    {
        if (intersects<TGeom>(geom, IntersectionMode::Any, LAYER_INCLUDED_PRIMARY_ORIGINAL)) {
            return true;
        }
        if (intersects<TGeom>(geom, IntersectionMode::All, LAYER_EXCLUDED_ORIGINAL)) {
            return false;
        }
        return intersects<TGeom>(geom, IntersectionMode::Any, LAYER_INCLUDED_SECONDARY_ORIGINAL);
    }

    template<typename TGeom>
    bool intersectsBuffered(const TGeom& geom) const
    {
        if (intersects<TGeom>(geom, IntersectionMode::Any, LAYER_INCLUDED_PRIMARY_BUFFERED)) {
            return true;
        }
        if (intersects<TGeom>(geom, IntersectionMode::All, LAYER_EXCLUDED_BUFFERED)) {
            return false;
        }
        return intersects<TGeom>(geom, IntersectionMode::Any, LAYER_INCLUDED_SECONDARY_BUFFERED);
    }

private:
    AreaOfInterest(const AreaOfInterest&) = delete;
    AreaOfInterest& operator=(const AreaOfInterest&) = delete;

    void addContourObjectById(
        revision::Snapshot& snapshot,
        revision::DBID contourId,
        const std::string& catId,
        Importance importance);

    void addRegionById(
        revision::Snapshot& snapshot,
        revision::DBID regionId);

    void buildLayer(
        const std::string& layerName,
        const DBID2PolygonVector& polygonGroups) const;

    template<typename TGeom>
    bool intersects(
        const TGeom& geom,
        IntersectionMode mode,
        const std::string& layerName) const
    {
        const auto& layer = (*coverage_)[layerName];
        return intersectsWithGeomPoints(geom, mode,
            [&layer](const geolib3::Point2& point) {
                return !layer.regions(point, boost::none).empty();
            });
    }

    double buffer_;

    DBID2PolygonVector includedPrimaryPolygonGroups_;
    DBID2PolygonVector includedPrimaryBufferedPolygonGroups_;

    DBID2PolygonVector excludedPolygonGroups_;
    DBID2PolygonVector excludedBufferedPolygonGroups_;

    DBID2PolygonVector includedSecondaryPolygonGroups_;
    DBID2PolygonVector includedSecondaryBufferedPolygonGroups_;

    boost::optional<geolib3::BoundingBox> boundingBox_;
    std::vector<geolib3::BoundingBox> bufferedBoundingBoxes_;

    std::string coverageDir_;
    std::unique_ptr<coverage5::Coverage> coverage_;
};

} // namespace validator
} // namespace wiki
} // namespace maps
