#pragma once

#include <yandex/maps/wiki/threadutils/threadedqueue.hpp>

#include <exception>
#include <memory>

namespace maps {
namespace wiki {
namespace validator {

template<typename T>
class AsyncResult
{
public:
    AsyncResult(T val)
        : val_(new T(std::move(val)))
    { }

    AsyncResult(std::exception_ptr exception)
        : exception_(std::move(exception))
    { }

    const T& value() const
    {
        rethrowIfException();
        return *val_;
    }

    T& value()
    {
        rethrowIfException();
        return *val_;
    }

    const std::exception_ptr& exception() const { return exception_; }

private:
    void rethrowIfException() const
    {
        if (exception_) {
            std::rethrow_exception(exception_);
        }
    }

    std::unique_ptr<T> val_;
    std::exception_ptr exception_;
};


template<typename T>
class AsyncResultsQueue
{
public:
    AsyncResultsQueue()
        : readyQueue_(std::make_shared<ReadyQueue>())
    { }

    class Task;

    using TaskType = std::function<T()>;

    Task task(TaskType function) const
    {
        return Task(readyQueue_, std::move(function));
    }

    std::list<AsyncResult<T>> popSome()
    {
        std::list<AsyncResult<T>> buffer;
        readyQueue_->popAll(buffer);
        return buffer;
    }

private:
    typedef ThreadedQueue<AsyncResult<T>> ReadyQueue;
    typedef std::shared_ptr<ReadyQueue> ReadyQueuePtr;

    ReadyQueuePtr readyQueue_;
};

template<typename T>
class AsyncResultsQueue<T>::Task
{
public:
    void operator()()
    {
        try {
            queue_->push(function_());
        } catch (...) {
            queue_->push(std::current_exception());
        }
    }

private:
    friend class AsyncResultsQueue<T>;

    Task(const ReadyQueuePtr& queue, TaskType function)
        : queue_(queue)
        , function_(std::move(function))
    { }

private:
    ReadyQueuePtr queue_;
    TaskType function_;
};

} // namespace validator
} // namespace wiki
} // namespace maps
