#include "category_revisions_loader.h"
#include "object_ids_loaders.h"
#include <maps/wikimap/mapspro/libs/validator/common/categories_list.h>
#include <maps/wikimap/mapspro/libs/validator/common/magic_strings.h>
#include <maps/wikimap/mapspro/libs/validator/common/name_relation_type_roles.h>

#include <yandex/maps/wiki/validator/categories.h>
#include <yandex/maps/wiki/configs/editor/attrdef.h>
#include <yandex/maps/wiki/configs/editor/categories.h>
#include <yandex/maps/wiki/configs/editor/magic_strings.h>
#include <yandex/maps/wiki/configs/editor/master_role.h>
#include <yandex/maps/wiki/configs/editor/topology_groups.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <maps/libs/log8/include/log8.h>

namespace maps {
namespace wiki {
namespace validator {

using namespace categories;

RevisionIdsLoadersStore::RevisionIdsLoadersStore(const configs::editor::ConfigHolder& editorConfig)
{
    insertCustomLoaders(editorConfig);
    insertLoadersFromConfig(editorConfig);

    bool hasCategoriesWithoutLoaders = false;
    for (auto loaderType : { LoaderType::LoadAll, LoaderType::LoadFromAoi, LoaderType::LoadBySelectedObjects }) {
        for (const auto& categoryId : allCategoryIds()) {
            if (!loadersByCategory_[loaderType].count(categoryId)) {
                ERROR() << "Category '" << categoryId << "' has no " << loaderType << " loader";
                hasCategoriesWithoutLoaders = true;
            }
        }
    }
    REQUIRE(!hasCategoriesWithoutLoaders, "Some categories haven't the full set of loaders");
}

bool RevisionIdsLoadersStore::hasLoader(const TCategoryId& categoryId, LoaderType loaderType) const
{
    auto loadersIt = loadersByCategory_.find(loaderType);
    ASSERT(loadersIt != loadersByCategory_.end());

    const auto& loaders = loadersIt->second;
    return loaders.find(categoryId) != loaders.end();
}

const RevisionIdsLoader& RevisionIdsLoadersStore::loader(const TCategoryId& categoryId, LoaderType loaderType) const
{
    auto loadersIt = loadersByCategory_.find(loaderType);
    ASSERT(loadersIt != loadersByCategory_.end());

    const auto& loaders = loadersIt->second;
    auto loader = loaders.find(categoryId);
    REQUIRE(loader != loaders.end(),
        "Revisions loader for category '" << categoryId << "' not found");
    return loader->second;
}

void RevisionIdsLoadersStore::insertLoader(TCategoryId categoryId, LoaderType loaderType, RevisionIdsLoader loader)
{
    checkCategoryExists(categoryId);
    auto result = loadersByCategory_[loaderType].emplace(categoryId, std::move(loader));
    DEBUG() << "Create loader for category '" << categoryId << "' result " << result.second;
}

void RevisionIdsLoadersStore::insertTransportThreadLoader(
    const TCategoryId& categoryId,
    const std::set<TCategoryId>& otherCategories)
{
    checkCategoryExists(categoryId);
    loadersByCategory_[LoaderType::LoadFromAoi].emplace(
        categoryId,
        createByRelatedIdsLoader(
            categoryId,
            otherCategories,
            { ASSIGNED_ROLE, ASSIGNED_THREAD_ROLE },
            RelationType::Slave));
}

void RevisionIdsLoadersStore::insertByRelatedIdsLoader(
    const TCategoryId& categoryId,
    LoaderType loaderType,
    const std::set<TCategoryId>& otherCategories,
    const std::set<std::string>& roleIds,
    RelationType relationType)
{
    checkCategoryExists(categoryId);
    loadersByCategory_[loaderType].emplace(
        categoryId,
        createByRelatedIdsLoader(
            categoryId,
            otherCategories,
            roleIds,
            relationType));
}

void RevisionIdsLoadersStore::insertCustomLoaders(const configs::editor::ConfigHolder& editorConfig)
{
    insertTransportThreadLoader(
        TRANSPORT_WATERWAY_ROUTE::id(),
        { TRANSPORT_WATERWAY_STOP::id(), TRANSPORT_WATERWAY_THREAD::id() });

    insertTransportThreadLoader(
        TRANSPORT_TRAM_ROUTE::id(),
        { TRANSPORT_STOP::id(), TRANSPORT_TRAM_THREAD::id() });

    insertTransportThreadLoader(
        TRANSPORT_BUS_ROUTE::id(),
        { TRANSPORT_STOP::id(), TRANSPORT_BUS_THREAD::id() });

    insertByRelatedIdsLoader(
        TRANSPORT_THREAD_STOP::id(),
        LoaderType::LoadFromAoi,
        {
            TRANSPORT_METRO_STATION::id(),
            TRANSPORT_STOP::id(),
            TRANSPORT_METRO_THREAD::id(),
            TRANSPORT_BUS_THREAD::id(),
            TRANSPORT_TRAM_THREAD::id(),
            TRANSPORT_WATERWAY_THREAD::id()
        },
        {
            ASSIGNED_THREAD_STOP_ROLE,
            PART_ROLE
        },
        RelationType::Master);

    insertLoader(
        TRANSPORT_THREAD_STOP::id(),
        LoaderType::LoadBySelectedObjects,
        loader(
            TRANSPORT_THREAD_STOP::id(),
            LoaderType::LoadFromAoi));

    insertByRelatedIdsLoader(
        TRANSPORT_TRANSITION::id(),
        LoaderType::LoadFromAoi,
        { TRANSPORT_METRO_STATION::id() },
        { STATION_A_ROLE, STATION_B_ROLE },
        RelationType::Slave);

    insertLoader(
        TRANSPORT_TRANSITION::id(),
        LoaderType::LoadBySelectedObjects,
        loader(
            TRANSPORT_TRANSITION::id(),
            LoaderType::LoadFromAoi));

    insertByRelatedIdsLoader(
        TRANSPORT_PASSAGEWAY::id(),
        LoaderType::LoadFromAoi,
        { TRANSPORT_METRO_STATION::id(), TRANSPORT_METRO_EXIT::id() },
        { STATION_ROLE, EXIT_ROLE },
        RelationType::Slave);

    insertLoader(
        TRANSPORT_PASSAGEWAY::id(),
        LoaderType::LoadBySelectedObjects,
        loader(
            TRANSPORT_PASSAGEWAY::id(),
            LoaderType::LoadFromAoi));

    insertByRelatedIdsLoader(
        TRANSPORT_TRANSITION_BOARDING::id(),
        LoaderType::LoadFromAoi,
        { TRANSPORT_METRO_STATION::id() },
        { STATION_PREVIOUS_ROLE },
        RelationType::Slave);

    insertLoader(
        TRANSPORT_TRANSITION_BOARDING::id(),
        LoaderType::LoadBySelectedObjects,
        loader(
            TRANSPORT_TRANSITION_BOARDING::id(),
            LoaderType::LoadFromAoi));

    insertByRelatedIdsLoader(
        TRANSPORT_PASSAGEWAY_BOARDING::id(),
        LoaderType::LoadFromAoi,
        { TRANSPORT_METRO_STATION::id() },
        { STATION_PREVIOUS_ROLE },
        RelationType::Slave);

    insertLoader(
        TRANSPORT_PASSAGEWAY_BOARDING::id(),
        LoaderType::LoadBySelectedObjects,
        loader(
            TRANSPORT_PASSAGEWAY_BOARDING::id(),
            LoaderType::LoadFromAoi));

    insertByRelatedIdsLoader(
        RD::id(),
        LoaderType::LoadFromAoi,
        { RD_EL::id(), ADDR::id() },
        { PART_ROLE, ASSOCIATED_WITH_ROLE },
        RelationType::Slave);

    insertLoader(RD::id(), LoaderType::LoadBySelectedObjects, createRdLoaderBySelectedObjectIds(editorConfig));

    insertLoader(AD::id(), LoaderType::LoadFromAoi, createAdmUnitLoader());
}

void RevisionIdsLoadersStore::insertLoadersFromConfig(const configs::editor::ConfigHolder& editorConfig)
{
    struct NameCategoryParams
    {
        std::set<std::string> roleIds;
        std::set<TCategoryId> masterCategoryIds;
    };

    std::map<TCategoryId, NameCategoryParams> nameCategoryParams;

    for (const auto& pair : editorConfig.categories()) {
        const auto& categoryId = pair.first;
        if (!isCategoryExisting(categoryId)) {
            continue;
        }

        insertLoader(
            categoryId,
            LoaderType::LoadAll,
            createAllRevisionIdsLoader(categoryId));

        const auto& category = pair.second;
        if (category.system()) {
            continue;
        }

        if (categoryGeomType(categoryId) != GeomType::None) {
            insertLoader(
                categoryId,
                LoaderType::LoadFromAoi,
                createInsideAoiRevisionIdsLoader(categoryId));
        }

        std::set<std::string> geomPartRoleIds;
        std::set<TCategoryId> geomPartCategoryIds;

        for (const auto& slaveRole : category.slavesRoles()) {
            if (slaveRole.tableRow()) {
                nameCategoryParams[slaveRole.categoryId()].roleIds.insert(slaveRole.roleId());
                nameCategoryParams[slaveRole.categoryId()].masterCategoryIds.insert(categoryId);
            }
            if (slaveRole.geomPart()) {
                if (isCategoryExisting(slaveRole.categoryId())) {
                    geomPartRoleIds.insert(slaveRole.roleId());
                    geomPartCategoryIds.insert(slaveRole.categoryId());
                }
            }
        }

        if (!geomPartCategoryIds.empty()) {
            insertByRelatedIdsLoader(
                categoryId,
                LoaderType::LoadFromAoi,
                geomPartCategoryIds,
                geomPartRoleIds,
                RelationType::Slave);
        }

        auto masterGeomPartRoleIds = category.masterRoleIds(configs::editor::roles::filters::IsGeom);

        std::set<TCategoryId> masterGeomPartCategoryIds;
        for (const auto& masterRoleId : masterGeomPartRoleIds) {
            for (const auto& masterRole : category.masterRole(masterRoleId)) {
                if (isCategoryExisting(masterRole.categoryId())) {
                    masterGeomPartCategoryIds.insert(masterRole.categoryId());
                }
            }
        }

        auto topoGroup = editorConfig.topologyGroups().findGroup(categoryId);
        if (topoGroup && categoryId == topoGroup->junctionsCategory()) {
            masterGeomPartRoleIds.insert(topoGroup->startJunctionRole());
            masterGeomPartRoleIds.insert(topoGroup->endJunctionRole());

            const auto& masterCategories = topoGroup->linearElementsCategories();
            masterGeomPartCategoryIds.insert(masterCategories.begin(), masterCategories.end());
        }

        if (!masterGeomPartCategoryIds.empty() || !geomPartCategoryIds.empty()) {
            insertLoader(
                categoryId,
                LoaderType::LoadBySelectedObjects,
                createLoaderBySelectedObjectIds(
                    editorConfig,
                    categoryId,
                    masterGeomPartCategoryIds,
                    masterGeomPartRoleIds));
        }

        if (!category.complex() && !hasLoader(categoryId, LoaderType::LoadBySelectedObjects)) {
            insertLoader(
                categoryId,
                LoaderType::LoadBySelectedObjects,
                createLoaderBySelectedObjectIds(
                    editorConfig,
                    categoryId,
                    {}, // no master categories
                    {})); // no master role ids
        }
    } // editorConfig.categories()

    for (const auto& pair : nameCategoryParams) {
        const auto& categoryId = pair.first;
        if (!isCategoryExisting(categoryId)) {
            continue;
        }

        const auto& params = pair.second;
        insertByRelatedIdsLoader(
            categoryId,
            LoaderType::LoadFromAoi,
            params.masterCategoryIds,
            params.roleIds,
            RelationType::Master);

        //copy loader
        insertLoader(
            categoryId,
            LoaderType::LoadBySelectedObjects,
            loader(
                categoryId,
                LoaderType::LoadFromAoi));
    }
}

} // namespace validator
} // namespace wiki
} // namespace maps
