#pragma once

#include "db_gateway.h"

#include <yandex/maps/wiki/validator/area_of_interest.h>
#include <yandex/maps/wiki/validator/message.h>

#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/spatial_relation.h>

#include <boost/optional.hpp>

namespace maps::wiki::validator {

class ImportantRegions
{
public:
    ImportantRegions(
        DBGateway& gateway,
        const AreaOfInterest& aoi);

    template<typename TGeom>
    RegionType determineRegionType(const TGeom& geom) const;

    template<typename TGeom>
    RegionType determineRegionType(const boost::optional<TGeom>& geom) const;

private:
    PolygonVector regions_;
};

template<typename TGeom>
RegionType ImportantRegions::determineRegionType(const TGeom& geom) const
{
    for (const auto& region : regions_) {
        if (geolib3::spatialRelation(region, geom, geolib3::SpatialRelation::Intersects)) {
            return RegionType::Important;
        }
    }
    return RegionType::Unimportant;
}

template<>
inline RegionType ImportantRegions::determineRegionType<boost::none_t>(const boost::none_t&) const
{
    return RegionType::Unimportant;
}

template<typename TGeom>
RegionType ImportantRegions::determineRegionType(const boost::optional<TGeom>& geom) const
{
    return geom ? determineRegionType(*geom) : RegionType::Unimportant;
}

} // namespace maps::wiki::validator
