#pragma once

#include "db_gateway.h"

#include <yandex/maps/wiki/validator/area_of_interest.h>
#include <yandex/maps/wiki/validator/categories.h>
#include <yandex/maps/wiki/validator/common.h>
#include <yandex/maps/wiki/validator/has_geom.h>
#include <yandex/maps/wiki/validator/message.h>

#include <yandex/maps/wiki/configs/editor/fwd.h>

#include <deque>
#include <type_traits>

namespace maps {
namespace wiki {
namespace validator {

struct ObjectDatum
{
    revision::ObjectRevision revision;

    Relations masters;
    Relations slaves;
};

template<class TObject>
class BaseObjectConstructor {

public:
    BaseObjectConstructor(
        std::list<Message>* messages,
        std::deque<TObject>* objects,
        ObjectIdToCommitId objectToCommitIds,
        ObjectIdSet deletedRelatedObjectIds,
        const AreaOfInterest& aoi,
        const configs::editor::Category* configCategory)
    : messages_(messages)
    , objects_(objects)
    , objectToCommitIds_(std::move(objectToCommitIds))
    , deletedRelatedObjectIds_(std::move(deletedRelatedObjectIds))
    , aoi_(aoi)
    , configCategory_(configCategory)
    {}

    const AreaOfInterest& aoi() const { return aoi_; }

    bool exists(TId objectId) const
    {
        return objectToCommitIds_.count(objectId) && !deleted(objectId);
    }

    bool deleted(TId objectId) const
    {
        return deletedRelatedObjectIds_.count(objectId);
    }

    RevisionID revisionId(TId objectId) const
    {
        const auto it = objectToCommitIds_.find(objectId);
        return it != objectToCommitIds_.end()
            ? RevisionID{objectId, it->second}
            : RevisionID::createNewID(objectId);
    }

    void report(
        Severity severity,
        const TCheckId& checkId,
        const std::string& message,
        RegionType regionType,
        const std::string& geom,
        const RevisionIds& revisionIds);

    template<class ...Args>
    void createObject(Args&&... args);

    void removeBadRelations(ObjectDatum& datum);
    void checkRelationsCardinality(const ObjectDatum& datum);

private:
    void removeRelationDuplicates(const RevisionID& revisionId, Relations& relations);
    void removeRelationsWithMissingObject(const RevisionID& revisionId, Relations& relations);

    std::list<Message>* messages_;
    std::deque<TObject>* objects_;
    ObjectIdToCommitId objectToCommitIds_;
    ObjectIdSet deletedRelatedObjectIds_;
    const AreaOfInterest& aoi_;
    const configs::editor::Category* configCategory_;
};

template<class TObject>
class SimpleObjectConstructor {

public:
    SimpleObjectConstructor(
        std::list<Message>* messages,
        std::deque<TObject>* objects,
        ObjectIdToCommitId objectToCommitIds,
        ObjectIdSet deletedRelatedObjectIds,
        const AreaOfInterest& aoi,
        const configs::editor::Category* configCategory)
    : base_(
        messages,
        objects,
        std::move(objectToCommitIds),
        std::move(deletedRelatedObjectIds),
        aoi,
        configCategory
    ) {}

    void construct(ObjectDatum datum);

private:
    BaseObjectConstructor<TObject> base_;
};

template<class TObject>
class GeomObjectConstructor {

public:
    GeomObjectConstructor(
        std::list<Message>* messages,
        std::deque<TObject>* objects,
        ObjectIdToCommitId objectToCommitIds,
        ObjectIdSet deletedRelatedObjectIds,
        const AreaOfInterest& aoi,
        const configs::editor::Category* configCategory)
    : base_(
        messages,
        objects,
        std::move(objectToCommitIds),
        std::move(deletedRelatedObjectIds),
        aoi,
        configCategory
    ) {}

    void construct(ObjectDatum datum);

private:
    BaseObjectConstructor<TObject> base_;
};

// does not load stray objects
template<class TObject>
class SystemObjectConstructor {
public:
    SystemObjectConstructor(
        std::list<Message>* messages,
        std::deque<TObject>* objects,
        ObjectIdToCommitId objectToCommitIds,
        ObjectIdSet deletedRelatedObjectIds,
        const AreaOfInterest& aoi,
        const configs::editor::Category* configCategory)
    : base_(
        messages,
        objects,
        std::move(objectToCommitIds),
        std::move(deletedRelatedObjectIds),
        aoi,
        configCategory
    ) {}

    void construct(ObjectDatum datum);

private:
    BaseObjectConstructor<TObject> base_;
};

template<class Category>
struct IsSystem {
    static const bool value =
        std::is_same<typename Category::TObject, Name>::value ||
        std::is_same<typename Category::TObject, Schedule>::value ||
        std::is_same<typename Category::TObject, FlatRange>::value;
};

template<class Category, class TObject = typename Category::TObject>
struct CategoryObjectConstructor
{
    typedef typename std::conditional<
        HasGeom<TObject>::value,
        GeomObjectConstructor<TObject>,
        typename std::conditional<
            IsSystem<Category>::value,
            SystemObjectConstructor<TObject>,
            SimpleObjectConstructor<TObject>
        >::type
    >::type type;
};

} // namespace validator
} // namespace wiki
} // namespace maps
