#include "object_ids_loaders.h"
#include "helper_loaders.h"
#include "data_source.h"

#include <yandex/maps/wiki/configs/editor/categories.h>

namespace maps {
namespace wiki {
namespace validator {

namespace {

using ObjectIdsLoaderFunc = std::function<ObjectIdSet(DataSource&)>;

/**
 * Returns object ids from selected object ids and loads object ids from geom part slaves
 */
struct ObjectIdsLoader
{
    TCategoryId categoryId;
    ObjectIdsLoaderFunc load;
};

struct GeomPartsInfo
{
    std::set<std::string> roleIds;
    std::vector<ObjectIdsLoader> loaders;
};

ObjectIdsLoader createObjectIdsLoader(
    const configs::editor::ConfigHolder& editorConfig,
    const TCategoryId& categoryId);

GeomPartsInfo computeGeomPartsInfo(
    const configs::editor::ConfigHolder& editorConfig,
    const TCategoryId& categoryId)
{
    GeomPartsInfo result;

    std::set<TCategoryId> geomPartCategoryIds;

    const auto& category = editorConfig.categories()[categoryId];
    for (const auto& slaveRole : category.slavesRoles()) {
        if (slaveRole.geomPart() && isCategoryExisting(slaveRole.categoryId())) {
            result.roleIds.insert(slaveRole.roleId());
            geomPartCategoryIds.insert(slaveRole.categoryId());
        }
    }

    result.loaders.reserve(geomPartCategoryIds.size());
    for (const auto& geomPartCategoryId : geomPartCategoryIds) {
        result.loaders.push_back(createObjectIdsLoader(editorConfig, geomPartCategoryId));
    }

    return result;
}

ObjectIdsLoader createObjectIdsLoader(
    const configs::editor::ConfigHolder& editorConfig,
    const TCategoryId& categoryId)
{
    ObjectIdsLoader loader;
    loader.categoryId = categoryId;
    loader.load =
        [categoryId, &editorConfig]
        (DataSource& dataSource) {
            auto geomPartsInfo = computeGeomPartsInfo(editorConfig, categoryId);

            auto resultObjectIds = dataSource.selectedObjectIds(categoryId);

            for (const auto& geomPartLoader : geomPartsInfo.loaders) {
                auto slaveObjectIds = geomPartLoader.load(dataSource);

                auto objectIds = loadObjectIdsByRelatedIds(
                    dataSource,
                    categoryId,
                    {slaveObjectIds.begin(), slaveObjectIds.end()},
                    RelationType::Slave,
                    geomPartsInfo.roleIds
                );
                resultObjectIds.insert(objectIds.begin(), objectIds.end());
            }

            return resultObjectIds;
        };

    return loader;
}

RevisionIds loadBySelectedObjectIds(
    DataSource& dataSource,
    const ObjectIdsLoader& loader,
    const std::set<TCategoryId>& masterCategories,
    const std::set<std::string>& masterRoleIds)
{
    ASSERT(dataSource.aoi().empty());
    ASSERT(dataSource.hasSelectedObjectIds());

    auto resultObjectIds = loader.load(dataSource);

    for (const auto& masterCategoryId : masterCategories) {
        const auto& masters = dataSource.collection(masterCategoryId);
        masters.checkLoaded();
        std::vector<TId> relatedIds = masters.objectIds();

        auto objectIds = loadObjectIdsByRelatedIds(
            dataSource,
            loader.categoryId,
            relatedIds,
            RelationType::Master,
            masterRoleIds
        );
        resultObjectIds.insert(objectIds.begin(), objectIds.end());
    }

    return revisionIdsByObjectIds(dataSource, resultObjectIds);
}

} // namespace

RevisionIdsLoader createLoaderBySelectedObjectIds(
    const configs::editor::ConfigHolder& editorConfig,
    const TCategoryId& categoryId,
    const std::set<TCategoryId>& masterCategories,
    const std::set<std::string>& masterRoleIds)
{
    RevisionIdsLoader loader;
    loader.dependencies = masterCategories;
    loader.load =
        [masterCategories, masterRoleIds, categoryId, &editorConfig]
        (DataSource& dataSource) {
            auto objectIdsLoader = createObjectIdsLoader(editorConfig, categoryId);

            return loadBySelectedObjectIds(
                dataSource,
                objectIdsLoader,
                masterCategories,
                masterRoleIds);
        };
    return loader;
}

RevisionIdsLoader createRdLoaderBySelectedObjectIds(
    const configs::editor::ConfigHolder& editorConfig)
{
    RevisionIdsLoader loader;

    loader.dependencies = { categories::ADDR::id() };

    loader.load = [&editorConfig](DataSource& dataSource) {
        auto objectIdsLoader = createObjectIdsLoader(editorConfig, categories::RD::id());

        auto resultObjectIds = objectIdsLoader.load(dataSource);

        const auto& addrs = dataSource.collection(categories::ADDR::id());
        addrs.checkLoaded();
        std::vector<TId> addrIds = addrs.objectIds();

        auto objectIds = loadObjectIdsByRelatedIds(
            dataSource,
            categories::RD::id(),
            addrIds,
            RelationType::Slave,
            { ASSOCIATED_WITH_ROLE }
        );
        resultObjectIds.insert(objectIds.begin(), objectIds.end());

        return revisionIdsByObjectIds(dataSource, resultObjectIds);
    };

    return loader;
}

} // namespace validator
} // namespace wiki
} // namespace maps
