#include "object_constructors.h"
#include <maps/wikimap/mapspro/libs/validator/common/exception.h>

#include <yandex/maps/wiki/configs/editor/master_role.h>
#include <yandex/maps/wiki/configs/editor/slave_role.h>
#include <yandex/maps/wiki/configs/editor/categories.h>

namespace maps {
namespace wiki {
namespace validator {

void checkSlaveRelationsCardinality(
    const configs::editor::Category& configCategory,
    const ObjectDatum& datum)
{
    std::unordered_map<std::string, std::vector<TId>> roleToRelatedSlaves;
    for (const auto& relation: datum.slaves) {
        roleToRelatedSlaves[relation.role].emplace_back(relation.other);
    }
    for (const auto& role: configCategory.slavesRoles()) {
        const auto& related = roleToRelatedSlaves[role.roleId()];
        if (related.size() < role.minOccurs()) {
            throw InvalidRelationsException(
                "not-enough-slave-relations-of-role-" + role.roleId(),
                datum.revision.id().objectId(),
                related,
                Severity::Critical);
        }
        if (related.size() > role.maxOccurs()) {
            throw InvalidRelationsException(
                "too-many-slave-relations-of-role-" + role.roleId(),
                datum.revision.id().objectId(),
                related,
                Severity::Critical);
        }
        roleToRelatedSlaves.erase(role.roleId());
    }

    for (const auto& pair : roleToRelatedSlaves) {
        throw InvalidRelationsException(
            "wrong-slave-relations-of-role-" + pair.first,
            datum.revision.id().objectId(),
            pair.second,
            Severity::Critical);
    }

    std::unordered_map<std::string, std::string> roleToOtherCategoryId;
    for (const auto& role: configCategory.slavesRoles()) {
        roleToOtherCategoryId.emplace(role.roleId(), role.categoryId());
    }
    for (const auto& relation: datum.slaves) {
        auto it = roleToOtherCategoryId.find(relation.role);
        if (it != roleToOtherCategoryId.end()
            && it->second != relation.otherCategoryId) {
            throw InvalidRelationsException(
                "wrong-slave-relative-category-" + relation.otherCategoryId,
                datum.revision.id().objectId(),
                {relation.other},
                Severity::Critical);
        }
    }
}

void checkMasterRelationsCardinality(
    const configs::editor::Category& configCategory,
    const ObjectDatum& datum)
{
    std::map<std::pair<std::string, std::string>, std::vector<TId>> roleAndOtherCategoryIdToRelatedMasters;
    for (const auto& relation: datum.masters) {
        roleAndOtherCategoryIdToRelatedMasters[{relation.role, relation.otherCategoryId}].emplace_back(relation.other);
    }
    for (const auto& role: configCategory.masterRoles()) {
        const auto& related = roleAndOtherCategoryIdToRelatedMasters[{role.roleId(), role.categoryId()}];
        if (related.size() < role.minOccurs()) {
            throw InvalidRelationsException(
                "not-enough-master-relations-of-role-" + role.roleId(),
                datum.revision.id().objectId(),
                related,
                Severity::Critical);
        }
        if (role.maxOccurs() && related.size() > *role.maxOccurs()) {
            throw InvalidRelationsException(
                "too-many-master-relations-of-role-" + role.roleId(),
                datum.revision.id().objectId(),
                related,
                Severity::Critical);
        }
        roleAndOtherCategoryIdToRelatedMasters.erase({role.roleId(), role.categoryId()});
    }

    for (const auto& pair : roleAndOtherCategoryIdToRelatedMasters) {
        throw InvalidRelationsException(
            "wrong-master-relations-of-role-" + pair.first.first,
            datum.revision.id().objectId(),
            pair.second,
            Severity::Critical);
    }
}

} // namespace validator
} // namespace wiki
} // namespace maps
