#pragma once

#include <yandex/maps/wiki/validator/common.h>
#include <yandex/maps/wiki/validator/objects/name.h>
#include <maps/wikimap/mapspro/libs/validator/common/exception.h>
#include <maps/wikimap/mapspro/libs/validator/common/magic_strings.h>
#include <maps/wikimap/mapspro/libs/validator/common/name_relation_type_roles.h>
#include <yandex/maps/wiki/validator/common/relation.h>

#include <algorithm>
#include <functional>
#include <stdexcept>
#include <vector>
#include <type_traits>

namespace maps::wiki::validator {

enum class IsMandatory { Yes, No };

inline std::vector<TId> extractRelations(
        const Relations& relations, const std::string& role);

inline std::vector<TId> extractRelations(
        const Relations& relations,
        const std::initializer_list<std::string>& roles);

inline TId extractRelation(
        const Relations& relations, const std::string& role,
        IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId);

inline TId extractRelation(
        const Relations& relations,
        const std::initializer_list<std::string>& roles,
        IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId);

inline std::vector<NameRelation> extractNameRelations(const Relations& slaves);

template<typename T>
T extractAttr(
        const AttrMap& attrs, const std::string& attr, IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId);

inline AttrMap::const_iterator findAttrBySuffix(
        const AttrMap& attrs, const std::string& suffix,
        const std::string& errorDescription, TId objectId);

template<class T>
T castAttr(
        const AttrMap& attrs, AttrMap::const_iterator iter, IsMandatory,
        const std::string& errorDescription, TId objectId);

template<typename T>
T extractAttrBySuffix(
        const AttrMap& attrs, const std::string& attr, IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId);

template<typename T>
std::optional<T> extractOptionalAttrBySuffix(const AttrMap& attrs, const std::string& attr,
        const std::string& errorDescription, TId objectId);

inline TFeatureType extractFeatureType(const AttrMap& attributes, TId objectId)
{
    return extractAttrBySuffix<int>(
            attributes, FT_TYPE_ATTR_SUFFIX, IsMandatory::No, "bad-ft-type", objectId);
}

// ---------------------- Implementation --------------------------

namespace {

inline TId extractRelationInternal(
        const Relations& relations,
        std::function<bool(const std::string&)> roleFilter,
        IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId)
{
    TId other = 0;
    for (const auto& rel : relations) {
        if (roleFilter(rel.role)) {
            if (other) {
                throw InvalidRelationsException(
                    errorDescription, objectId,
                    {other, rel.other});
            }
            other = rel.other;
        }
    }

    if (isMandatory == IsMandatory::Yes) {
        requireOnObjectLoad(other != 0, errorDescription, objectId);
    }

    return other;
}

inline std::vector<TId> extractRelationsInternal(
        const Relations& relations,
        std::function<bool(const std::string&)> roleFilter)
{
    std::vector<TId> ret;
    for (const auto& rel : relations) {
        if (roleFilter(rel.role)) {
            ret.push_back(rel.other);
        }
    }
    ret.shrink_to_fit();
    return ret;
}

} // namespace

inline TId extractRelation(
        const Relations& relations, const std::string& role,
        IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId)
{
    return extractRelationInternal(
        relations, [&role](const std::string& r){ return role == r; },
        isMandatory,
        errorDescription, objectId);
}

inline TId extractRelation(
        const Relations& relations,
        const std::initializer_list<std::string>& roles,
        IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId)
{
    return extractRelationInternal(
        relations,
        [&roles](const std::string& r)
            { return std::count(std::begin(roles), std::end(roles), r); },
        isMandatory,
        errorDescription, objectId);
}

inline std::vector<TId> extractRelations(
        const Relations& relations, const std::string& role)
{
    return extractRelationsInternal(
        relations, [&role](const std::string& r){ return role == r; });
}

inline std::vector<TId> extractRelations(
        const Relations& relations,
        const std::initializer_list<std::string>& roles)
{
    return extractRelationsInternal(
        relations,
        [&roles](const std::string& r)
            { return std::count(std::begin(roles), std::end(roles), r); });
}

inline std::vector<NameRelation> extractNameRelations(const Relations& slaves)
{
    std::vector<NameRelation> result;
    for (const auto& slaveRel : slaves) {
        auto nameRelTypeIt = ROLE_TO_NAME_RELATION_TYPE.find(slaveRel.role);
        if (nameRelTypeIt != std::end(ROLE_TO_NAME_RELATION_TYPE)) {
            result.emplace_back(nameRelTypeIt->second, slaveRel.other);
        }
    }
    result.shrink_to_fit();
    return result;
}

template<class T>
T castAttr(
        const AttrMap& attrs, AttrMap::const_iterator iter, IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId)
{
    static_assert(std::is_enum<T>(), "T is not an enumeration type");
    int attrVal = castAttr<int>(
            attrs, iter, isMandatory, errorDescription, objectId);
    return static_cast<T>(attrVal);
}

template<>
inline std::string castAttr<std::string>(
        const AttrMap& attrs, AttrMap::const_iterator iter, IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId)
{
    if (isMandatory == IsMandatory::No && iter == attrs.end()) {
        return std::string();
    }
    requireOnObjectLoad(iter != attrs.end(), errorDescription, objectId);
    return iter->second;
}

template<>
inline int castAttr<int>(
        const AttrMap& attrs, AttrMap::const_iterator iter, IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId)
{
    if (isMandatory == IsMandatory::No && iter == attrs.end()) {
        return 0;
    }
    requireOnObjectLoad(iter != attrs.end(), errorDescription, objectId);

    try {
        size_t pos = -1;
        int value = std::stoi(iter->second, &pos);
        requireOnObjectLoad(
            pos == iter->second.size(), errorDescription, objectId);
        return value;
    } catch (const std::logic_error&) {
        throw ObjectLoadingException(errorDescription, objectId);
    }
}

template<>
inline uint16_t castAttr<uint16_t>(
        const AttrMap& attrs, AttrMap::const_iterator iter, IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId)
{
    if (isMandatory == IsMandatory::No && iter == attrs.end()) {
        return 0;
    }
    requireOnObjectLoad(iter != attrs.end(), errorDescription, objectId);

    try {
        size_t pos = -1;
        uint16_t value = std::stoul(iter->second, &pos);
        requireOnObjectLoad(
            pos == iter->second.size(), errorDescription, objectId);
        return value;
    } catch (const std::logic_error&) {
        throw ObjectLoadingException(errorDescription, objectId);
    }
}


template<>
inline double castAttr<double>(
        const AttrMap& attrs, AttrMap::const_iterator iter, IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId)
{
    if (isMandatory == IsMandatory::No && iter == attrs.end()) {
        return 0;
    }
    requireOnObjectLoad(iter != attrs.end(), errorDescription, objectId);

    try {
        size_t pos = -1;
        double value = std::stod(iter->second, &pos);
        requireOnObjectLoad(
            pos == iter->second.size(), errorDescription, objectId);
        return value;
    } catch (const std::logic_error&) {
        throw ObjectLoadingException(errorDescription, objectId);
    }
}

template<>
inline bool castAttr<bool>(
        const AttrMap& attrs, AttrMap::const_iterator iter, IsMandatory,
        const std::string& errorDescription, TId objectId)
{
    requireOnObjectLoad(
            iter == attrs.end() || iter->second == "1",
            errorDescription, objectId);
    return iter != attrs.end();
}

inline AttrMap::const_iterator findAttrBySuffix(
        const AttrMap& attrs, const std::string& suffix,
        const std::string& errorDescription, TId objectId)
{
    auto ret = attrs.end();

    for (auto iter = attrs.begin(); iter != attrs.end(); ++iter) {
        if (iter->first.ends_with(suffix)) {
            requireOnObjectLoad(ret == attrs.end(), errorDescription, objectId);
            ret = iter;
        }
    }

    return ret;
}

template<typename T>
T extractAttrBySuffix(
        const AttrMap& attrs, const std::string& suffix, IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId)
{
    auto ret = findAttrBySuffix(attrs, suffix, errorDescription, objectId);
    return castAttr<T>(attrs, ret, isMandatory, errorDescription, objectId);
}

template<typename T>
std::optional<T> extractOptionalAttrBySuffix(const AttrMap& attrs, const std::string& suffix,
    const std::string& errorDescription, TId objectId)
{
    auto ret = findAttrBySuffix(attrs, suffix, errorDescription, objectId);
    if (ret == attrs.end()) {
        return std::nullopt;
    }
    return castAttr<T>(attrs, ret, IsMandatory::No, errorDescription, objectId);
}

template<typename T>
T extractAttr(
        const AttrMap& attrs, const std::string& attr, IsMandatory isMandatory,
        const std::string& errorDescription, TId objectId)
{
    return castAttr<T>(
            attrs, attrs.find(attr), isMandatory,
            errorDescription, objectId);
}

inline std::string extractImportSourceId(const AttrMap& attrs, TId objectId)
{
    return extractAttr<std::string>(
        attrs, IMPORT_SOURCE_ID_ATTR, IsMandatory::No, "bad-import_source_id", objectId);
}

inline std::string extractImportSource(const AttrMap& attrs, TId objectId)
{
    return extractAttr<std::string>(
        attrs, IMPORT_SOURCE_ATTR, IsMandatory::No, "bad-import_source", objectId);
}

inline std::string extractGeobaseId(const AttrMap& attrs, TId objectId)
{
    return extractAttr<std::string>(
        attrs, GEOBASE_ID, IsMandatory::No, "bad-geobase_id", objectId);
}

} // namespace maps::wiki::validator
