#include <yandex/maps/wiki/validator/objects/poi.h>

#include "object_init_helpers.h"
#include <yandex/maps/wiki/validator/categories.h>
#include <maps/wikimap/mapspro/libs/validator/common/exception.h>
#include <maps/wikimap/mapspro/libs/validator/common/magic_strings.h>
#include <maps/libs/enum_io/include/enum_io.h>

namespace maps::wiki::validator {

namespace {

const std::initializer_list<std::string> INDOOR_POI_ROLES = {
    ASSIGNED_INDOOR_POI_AUTO_ROLE,
    ASSIGNED_INDOOR_POI_EDU_ROLE,
    ASSIGNED_INDOOR_POI_FINANCE_ROLE,
    ASSIGNED_INDOOR_POI_FOOD_ROLE,
    ASSIGNED_INDOOR_POI_GOVERMENT_ROLE,
    ASSIGNED_INDOOR_POI_LEISURE_ROLE,
    ASSIGNED_INDOOR_POI_MEDICINE_ROLE,
    ASSIGNED_INDOOR_POI_RELIGION_ROLE,
    ASSIGNED_INDOOR_POI_SERVICE_ROLE,
    ASSIGNED_INDOOR_POI_SHOPPING_ROLE,
    ASSIGNED_INDOOR_POI_SPORT_ROLE
};

const enum_io::Representations<Poi::Location> POI_LOCATION_ENUM_REPR {
    {Poi::Location::Empty, POI_LOCATION_EMPTY},
    {Poi::Location::EntranceSign, POI_LOCATION_ENTRANCE_SIGN},
    {Poi::Location::Indoor, POI_LOCATION_INDOOR}
};

const enum_io::Representations<Poi::PositionQuality> POI_POSITION_QUALITY_ENUM_REPR {
    {Poi::PositionQuality::Empty, POI_POSITION_QUALITY_EMPTY},
    {Poi::PositionQuality::Auto, POI_POSITION_QUALITY_AUTO},
    {Poi::PositionQuality::Undefined, POI_POSITION_QUALITY_UNDEFINED},
    {Poi::PositionQuality::Approximate, POI_POSITION_QUALITY_APPROXIMATE},
    {Poi::PositionQuality::User, POI_POSITION_QUALITY_USER},
    {Poi::PositionQuality::Precise, POI_POSITION_QUALITY_PRECISE}
};

const std::string MESSAGE_BAD_BUSINESS_ID = "bad-business-id";
const std::string MESSAGE_BAD_BUSINESS_RUBRIC_ID = "bad-business-rubric-id";
const std::string MESSAGE_BAD_IS_GEOPRODUCT = "bad-is-geoproduct";
const std::string MESSAGE_BAD_ASSIGNED_RELATION = "bad-assigned-relation";
const std::string MESSAGE_BAD_IMPORT_SOURCE = "bad-import-source";
const std::string MESSAGE_BAD_IMPORT_SOURCE_ID = "bad-import-source-id";
const std::string MESSAGE_BAD_POI_LOCATION = "bad-poi-location";
const std::string MESSAGE_BAD_POI_POSITION_QUALITY = "bad-poi-position-quality";

} // namespace

DEFINE_ENUM_IO(Poi::Location, POI_LOCATION_ENUM_REPR);
DEFINE_ENUM_IO(Poi::PositionQuality, POI_POSITION_QUALITY_ENUM_REPR);

Poi::Poi(
        TId id,
        TGeom geom,
        const AttrMap& attributes,
        const Relations& masters,
        const Relations& slaves)
    : id_(id)
    , featureType_(extractFeatureType(attributes, id))
    , geom_(std::move(geom))
    , parent_(
        extractRelation(
            masters,
            INDOOR_POI_ROLES,
            IsMandatory::No,
            MESSAGE_BAD_PARENT_RELATION,
            id))
    , assignedObjects_()
    , names_(extractNameRelations(slaves))
    , businessId_(
        extractAttr<std::string>(
            attributes, BUSINESS_ID_ATTR_NAME, IsMandatory::No,
            MESSAGE_BAD_BUSINESS_ID, id))
    , businessRubricId_(
        extractAttr<std::string>(
            attributes, BUSINESS_RUBRIC_ID_ATTR_NAME, IsMandatory::No,
            MESSAGE_BAD_BUSINESS_RUBRIC_ID, id))
    , isGeoproduct_(
        extractAttrBySuffix<bool>(
            attributes, IS_GEOPRODUCT_ATTR_SUFFIX, IsMandatory::No,
            MESSAGE_BAD_IS_GEOPRODUCT, id))
    , importSource_(
        extractAttr<std::string>(
            attributes, IMPORT_SOURCE_ATTR, IsMandatory::No,
            MESSAGE_BAD_IMPORT_SOURCE, id))
    , importSourceId_(
        extractAttr<std::string>(
            attributes, IMPORT_SOURCE_ID_ATTR, IsMandatory::No,
            MESSAGE_BAD_IMPORT_SOURCE_ID, id))
    , location_(
        maps::enum_io::fromString<Poi::Location>(extractAttrBySuffix<std::string>(
            attributes, POI_LOCATION_ATTR_SUFFIX, IsMandatory::No,
            MESSAGE_BAD_POI_LOCATION, id)))
    , positionQuality_(
        maps::enum_io::fromString<Poi::PositionQuality>(extractAttrBySuffix<std::string>(
            attributes, POI_POSITION_QUALITY_ATTR_SUFFIX, IsMandatory::No,
            MESSAGE_BAD_POI_LOCATION, id)))
{
    auto checkRelation = [this](const Relation& slaveRel,
                                TCategoryId expectedCategoryId) {
        if (slaveRel.otherCategoryId != expectedCategoryId) {
            throw InvalidRelationsException(
                MESSAGE_BAD_ASSIGNED_RELATION, id_, {slaveRel.other});
        }
    };

    for (const auto& slaveRel : slaves) {
        if (slaveRel.role == BLD_ASSIGNED_ROLE) {
            checkRelation(slaveRel, categories::BLD::id());
            assignedObjects_.push_back(Poi::AssignedObject{
                slaveRel.other, Poi::AssignedObject::Type::Building});
        } else if (slaveRel.role == URBAN_AREAL_ASSIGNED_ROLE) {
            checkRelation(slaveRel, categories::URBAN_AREAL::id());
            assignedObjects_.push_back(Poi::AssignedObject{
                slaveRel.other, Poi::AssignedObject::Type::UrbanAreal});
        } else if (slaveRel.role == ENTRANCE_ASSIGNED_ROLE) {
            checkRelation(slaveRel, categories::POI_ENTRANCE::id());
            assignedObjects_.push_back(Poi::AssignedObject{
                slaveRel.other, Poi::AssignedObject::Type::Entrance});
        }
    }
}

} // namespace maps::wiki::validator

