#include <yandex/maps/wiki/validator/objects/road_element.h>

#include "object_init_helpers.h"
#include <maps/wikimap/mapspro/libs/validator/common/exception.h>
#include <maps/wikimap/mapspro/libs/validator/common/magic_strings.h>

#include <maps/libs/common/include/exception.h>

#include <boost/lexical_cast.hpp>

namespace maps::wiki::validator {

namespace {

const common::Lanes EMPTY_LANES;
const std::string EMPTY_STRING;

std::unique_ptr<common::Lanes> makeLanes(const std::string& lanesStr)
{
    auto lanes = common::lanesFromString(lanesStr);
    if (lanes.empty()) {
        return {};
    }
    return std::make_unique<common::Lanes>(std::move(lanes));
}

} // namespace

RoadElement::RoadElement(
        TId id,
        TGeom geom,
        const AttrMap& attributes,
        const Relations& masters,
        const Relations& slaves)
    : id_(id)
    , geom_(std::move(geom))
    , parents_(extractRelations(masters, PART_ROLE))
    , speedLimit_(
        extractOptionalAttrBySuffix<int>(
            attributes, SPEED_LIMIT_ATTR_SUFFIX,
            "bad-speed-limit", id))
    , speedLimitT_(
        extractOptionalAttrBySuffix<int>(
            attributes, SPEED_LIMIT_T_ATTR_SUFFIX,
            "bad-speed-limit-t", id))
    , speedLimitTruck_(
        extractOptionalAttrBySuffix<int>(
            attributes, SPEED_LIMIT_TRUCK_ATTR_SUFFIX,
            "bad-speed-limit-truck", id))
    , speedLimitTruckT_(
        extractOptionalAttrBySuffix<int>(
            attributes, SPEED_LIMIT_TRUCK_T_ATTR_SUFFIX,
            "bad-speed-limit-truck-t", id))

    , fromLanes_(
        makeLanes(
            extractAttrBySuffix<std::string>(
                attributes, F_LANE_ATTR_SUFFIX, IsMandatory::No,
                "bad-f-lane-attribute", id)))
    , toLanes_(
        makeLanes(
            extractAttrBySuffix<std::string>(
                attributes, T_LANE_ATTR_SUFFIX, IsMandatory::No,
                "bad-t-lane-attribute", id)))
    , startJunction_(
        extractRelation(
            slaves, START_JUNCTION_ROLE, IsMandatory::Yes,
            "bad-start-junction-relation", id))
    , endJunction_(
        extractRelation(
            slaves, END_JUNCTION_ROLE, IsMandatory::Yes,
            "bad-end-junction-relation", id))
    , fc_(
        extractAttr<int>(
            attributes, FC_ATTR_NAME, IsMandatory::No,
            "bad-fc", id))
    , fromZlevel_(
        extractAttr<int>(
            attributes, F_ZLEV_ATTR_NAME, IsMandatory::No,
            "bad-f-zlev", id))
    , toZlevel_(
        extractAttr<int>(
            attributes, T_ZLEV_ATTR_NAME, IsMandatory::No,
            "bad-t-zlev", id))
    , accessId_(
        extractAttr<common::AccessId>(
            attributes, ACCESS_ID_ATTR_NAME, IsMandatory::No,
            "bad-access-id", id))
    , direction_(Direction::Both)
    , fow_(
        extractAttr<common::FOW>(
            attributes, FOW_ATTR_NAME, IsMandatory::No,
            "bad-fow", id))
    , structType_(
        extractAttr<common::StructType>(
            attributes, STRUCT_TYPE_ATTR_NAME, IsMandatory::No,
            "bad-struct-type", id))
    , paved_(
        extractAttrBySuffix<bool>(
            attributes, PAVED_ATTR_SUFFIX, IsMandatory::No,
            "bad-paved-attribute", id))
    , residential_(
        extractAttrBySuffix<bool>(
            attributes, RESIDENTIAL_ATTR_SUFFIX, IsMandatory::No,
            "bad-residential-attribute", id))
    , backBus_(
        extractAttrBySuffix<bool>(
            attributes, BACK_BUS_ATTR_SUFFIX, IsMandatory::No,
            "bad-back-bus-attribute", id))
    , backTaxi_(
        extractAttrBySuffix<bool>(
            attributes, BACK_TAXI_ATTR_SUFFIX, IsMandatory::No,
            "bad-back-taxi-attribute", id))
    , backBicycle_(
        extractAttrBySuffix<bool>(
            attributes, BACK_BICYCLE_ATTR_SUFFIX, IsMandatory::No,
            "bad-back-bicycle-attribute", id))
    , dr_(
        extractAttrBySuffix<bool>(
            attributes, DR_ATTR_SUFFIX, IsMandatory::No,
            "bad-dr-attribute", id))
    , underConstruction_(
        extractAttrBySuffix<bool>(
            attributes, SRV_UC_ATTR_SUFFIX, IsMandatory::No,
            "bad-srv-uc-attribute", id))
    , toll_(
        extractAttrBySuffix<bool>(
            attributes, TOLL_ATTR_SUFFIX, IsMandatory::No,
            "bad-toll-attribute", id))
    , noScooterAccess_(
        extractAttrBySuffix<bool>(
            attributes, NO_SCOOTER_ACCESS_ATTR_SUFFIX, IsMandatory::No,
            "bad-no-scooter-access-attribute", id))
    , noMotoAccess_(
        extractAttrBySuffix<bool>(
            attributes, NO_MOTO_ACCESS_ATTR_SUFFIX, IsMandatory::No,
            "bad-no-moto-access-attribute", id))
{
    requireOnObjectLoad(common::isValid(accessId_), "bad-access-id", id);
    requireOnObjectLoad(common::isValid(fow_), "bad-fow", id);
    requireOnObjectLoad(common::isValid(structType_), "bad-struct-type", id);

    auto onewayAttrValue = extractAttr<std::string>(
        attributes, ONEWAY_ATTR_NAME, IsMandatory::No,
        "bad-oneway", id);
    if (!onewayAttrValue.empty()) {
        try {
            direction_ = boost::lexical_cast<Direction>(onewayAttrValue);
        } catch (boost::bad_lexical_cast&) {
            throw ObjectLoadingException("bad-oneway", id);
        }
    }
    requireOnObjectLoad(geom_.pointsNumber() >= 2, MESSAGE_BAD_GEOMETRY, id);

    auto openAtStr =
        extractAttrBySuffix<std::string>(
            attributes, SRV_UC_OPEN_AT_ATTR_SUFFIX, IsMandatory::No,
            "bad-srv-uc-open-at-attribute", id);

    if (!openAtStr.empty()) {
        openAt_ = std::unique_ptr<std::string>(new std::string(std::move(openAtStr)));
    }
}

const common::Lanes& RoadElement::fromLanes() const
{
    return fromLanes_ ? *fromLanes_ : EMPTY_LANES;
}

const common::Lanes& RoadElement::toLanes() const
{
    return toLanes_ ? *toLanes_ : EMPTY_LANES;
}

const std::string& RoadElement::openAt() const
{
    return openAt_ ? *openAt_ : EMPTY_STRING;
}

} // namespace maps::wiki::validator
