#include <yandex/maps/wiki/validator/objects/schedule.h>

#include "object_init_helpers.h"
#include <maps/wikimap/mapspro/libs/validator/common/exception.h>
#include <maps/wikimap/mapspro/libs/validator/common/magic_strings.h>
#include <yandex/maps/wiki/validator/categories.h>

namespace maps::wiki::validator {

namespace {
const std::string MESSAGE_BAD_SCHEDULE_RELATION = "bad-schedule-relation";
const std::string MESSAGE_BAD_WEEKDAYS_MASK = "bad-weekdays-mask";
} // namespace

Schedule::Schedule(
        TId id,
        const AttrMap& attributes,
        const Relations& masters,
        const Relations& /* slaves */)
    : id_(id)
    , weekdays_(common::WeekdayFlags::None)
    , affectedObject_{0, AffectedObject::Type::Unknown}
    , alternative_(false)
{
    for (const auto& masterRel : masters) {
        if (masterRel.role == APPLIED_TO_ROLE) {
            requireOnObjectLoad(
                !affectedObject_.id,
                MESSAGE_BAD_SCHEDULE_RELATION, affectedObject_.id);
            affectedObject_.id = masterRel.other;

            if (masterRel.otherCategoryId == categories::COND::id()) {
                affectedObject_.type = AffectedObject::Type::Condition;
            } else if (masterRel.otherCategoryId == categories::COND_CAM::id()) {
                affectedObject_.type = AffectedObject::Type::Camera;
            } else if (masterRel.otherCategoryId == categories::COND_LANE::id()) {
                affectedObject_.type = AffectedObject::Type::CondLane;
            } else if (attributes.count(CAT_PREFIX + categories::FREQ_DT::id())) {
                affectedObject_.type = AffectedObject::Type::TransportThread;
            } else if (attributes.count(CAT_PREFIX + categories::VEHICLE_RESTRICTION::id())) {
                affectedObject_.type = AffectedObject::Type::VehicleRestriction;
            } else {
                affectedObject_.type = AffectedObject::Type::Unknown;
            }
        }
    }

    TId parentId = affectedObject_.id;
    if (!parentId) {
        // unbound object, do not check
        return;
    }

    timeStart_ = extractAttrBySuffix<std::string>(
        attributes, SCHEDULE_TIME_START_ATTR_SUFFIX, IsMandatory::No, {}, parentId);
    timeEnd_ = extractAttrBySuffix<std::string>(
        attributes, SCHEDULE_TIME_END_ATTR_SUFFIX, IsMandatory::No, {}, parentId);

    departureTime_ = extractAttrBySuffix<std::string>(
        attributes, SCHEDULE_DEPARTURE_TIME_ATTR_SUFFIX, IsMandatory::No, {}, parentId);

    dateStart_ = extractAttrBySuffix<std::string>(
        attributes, SCHEDULE_DATE_START_ATTR_SUFFIX, IsMandatory::No, {}, parentId);
    dateEnd_ = extractAttrBySuffix<std::string>(
        attributes, SCHEDULE_DATE_END_ATTR_SUFFIX, IsMandatory::No, {}, parentId);

    weekdays_ = extractAttrBySuffix<common::WeekdayFlags>(
        attributes, SCHEDULE_DAY_ATTR_SUFFIX, IsMandatory::Yes,
        MESSAGE_BAD_WEEKDAYS_MASK, parentId);
    requireOnObjectLoad(
        weekdays_ >= common::WeekdayFlags::Min && weekdays_ <= common::WeekdayFlags::Max,
        MESSAGE_BAD_WEEKDAYS_MASK, parentId);

    frequency_ = extractAttrBySuffix<int>(
        attributes, SCHEDULE_FREQ_ATTR_SUFFIX, IsMandatory::No, {}, parentId);

    alternative_ = extractAttrBySuffix<bool>(
        attributes, SCHEDULE_ALTERNATIVE_ATTR_SUFFIX, IsMandatory::No, {}, parentId);
}

} // namespace maps::wiki::validator
