#include <yandex/maps/wiki/validator/objects/transport_boarding.h>

#include "object_init_helpers.h"
#include <maps/wikimap/mapspro/libs/validator/common/exception.h>
#include <maps/wikimap/mapspro/libs/validator/common/magic_strings.h>
#include <maps/libs/enum_io/include/enum_io.h>

#include <boost/algorithm/string/split.hpp>

namespace maps::wiki::validator {

namespace {

const std::string MESSAGE_BAD_BOARDING_REL = "bad-boarding-relation";
const std::string MESSAGE_BAD_STATION_PREVIOUS_REL = "bad-previous-station-previous-relation";
const std::string MESSAGE_BAD_STATION_NEXT_REL = "bad-previous-station-next-relation";
const std::string MESSAGE_BAD_ONEWAY = "bad-oneway";
const std::string MESSAGE_BAD_BOARDING_OPTIONS = "bad-boarding-options";

const enum_io::Representations<BoardingOptions> BOARDING_OPTIONS_ENUM_REPR {
    {BoardingOptions::Head, BOARDING_OPTIONS_HEAD},
    {BoardingOptions::MiddleHead, BOARDING_OPTIONS_MIDDLE_HEAD},
    {BoardingOptions::Middle, BOARDING_OPTIONS_MIDDLE},
    {BoardingOptions::MiddleTail, BOARDING_OPTIONS_MIDDLE_TAIL},
    {BoardingOptions::Tail, BOARDING_OPTIONS_TAIL}
};

std::set<BoardingOptions>
parseBoardingOptions(const std::string& attrValue)
{
    std::vector<std::string> parts;
    boost::split(parts, attrValue,
        [] (char c) { return c == MULTIVALUE_ATTR_DELIMITER; });

    std::set<BoardingOptions> result;
    for (const auto& part : parts) {
        result.insert(maps::enum_io::fromString<BoardingOptions>(part));
    }
    return result;
}

} // namespace

DEFINE_ENUM_IO(BoardingOptions, BOARDING_OPTIONS_ENUM_REPR);

TransportPassagewayBoarding::TransportPassagewayBoarding(
        TId id,
        const AttrMap& attributes,
        const Relations& masters,
        const Relations& slaves)
    : id_(id)
    , passageway_(extractRelation(masters, BOARDING_ROLE,
        IsMandatory::Yes, MESSAGE_BAD_BOARDING_REL, id))
    , previousStation_(extractRelation(slaves, STATION_PREVIOUS_ROLE,
        IsMandatory::Yes, MESSAGE_BAD_STATION_PREVIOUS_REL, id))
    , boardingOptions_(
        parseBoardingOptions(extractAttrBySuffix<std::string>(
            attributes, BOARDING_OPTIONS_ATTR_SUFFIX, IsMandatory::Yes,
            MESSAGE_BAD_BOARDING_OPTIONS, id)))
{
}

TransportTransitionBoarding::TransportTransitionBoarding(
        TId id,
        const AttrMap& attributes,
        const Relations& masters,
        const Relations& slaves)
    : id_(id)
    , transition_(extractRelation(masters, BOARDING_ROLE,
        IsMandatory::Yes, MESSAGE_BAD_BOARDING_REL, id))
    , previousStation_(extractRelation(slaves, STATION_PREVIOUS_ROLE,
        IsMandatory::Yes, MESSAGE_BAD_STATION_PREVIOUS_REL, id))
    , nextStation_(extractRelation(slaves, STATION_NEXT_ROLE,
        IsMandatory::No, MESSAGE_BAD_STATION_NEXT_REL, id))
    , boardingOptions_(
        parseBoardingOptions(extractAttrBySuffix<std::string>(
            attributes, BOARDING_OPTIONS_ATTR_SUFFIX, IsMandatory::Yes,
            MESSAGE_BAD_BOARDING_OPTIONS, id)))
{
    auto iterOneway =
        findAttrBySuffix(attributes, ONEWAY_ATTR_SUFFIX, MESSAGE_BAD_ONEWAY, id);
    if (iterOneway != attributes.end()) {
        const auto& value = iterOneway->second;
        if (value == "F") {
            oneway_ = Oneway::From;
        } else if (value == "T") {
            oneway_ = Oneway::To;
        } else {
            requireOnObjectLoad(false, MESSAGE_BAD_ONEWAY, id);
        }
    }
}

} // namespace maps::wiki::validator
