#include <yandex/maps/wiki/validator/objects/vehicle_restriction.h>

#include "object_init_helpers.h"
#include <maps/wikimap/mapspro/libs/validator/common/exception.h>
#include <maps/wikimap/mapspro/libs/validator/common/magic_strings.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>

#include <algorithm>

namespace maps::wiki::validator {

namespace {
const std::unordered_set<std::string> ALL_PARAMETERS
{
    VEHICLE_RESTRICTION_WEIGHT_LIMIT_ATTR,
    VEHICLE_RESTRICTION_AXLE_WEIGHT_LIMIT_ATTR,
    VEHICLE_RESTRICTION_MAX_WEIGHT_LIMIT_ATTR,
    VEHICLE_RESTRICTION_HEIGHT_LIMIT_ATTR,
    VEHICLE_RESTRICTION_WIDTH_LIMIT_ATTR,
    VEHICLE_RESTRICTION_LENGTH_LIMIT_ATTR,
    VEHICLE_RESTRICTION_PAYLOAD_LIMIT_ATTR,
    VEHICLE_RESTRICTION_MIN_ECO_CLASS_ATTR,
    VEHICLE_RESTRICTION_TRAILER_NOT_ALLOWED_ATTR
};

template<typename ParamType>
std::optional<ParamType>
 readParameter(const AttrMap& attributes, const std::string& attribute, TId objectId)
{
    auto it = attributes.find(attribute);
    if (it == attributes.end()) {
        return std::nullopt;
    }
    return std::optional<ParamType>(
        castAttr<ParamType>(attributes, it, IsMandatory::No, "bad-" + attribute, objectId));
}

template<>
std::optional<EcoClass>
readParameter(const AttrMap& attributes, const std::string& attribute, TId objectId)
{
    auto it = attributes.find(attribute);
    if (it == attributes.end()) {
        return std::nullopt;
    }
    auto ecoClass = maps::enum_io::tryFromString<EcoClass>(it->second);
    if (!ecoClass) {
        throw ObjectLoadingException("bad-eco-class-value", objectId);
    }
    return ecoClass;
}
} // namespace

constexpr enum_io::Representations<EcoClass> ECOCLASS_ENUM_REPRESENTATION {
    {EcoClass::Euro0, "euro0"},
    {EcoClass::Euro1, "euro1"},
    {EcoClass::Euro2, "euro2"},
    {EcoClass::Euro3, "euro3"},
    {EcoClass::Euro4, "euro4"},
    {EcoClass::Euro5, "euro5"},
    {EcoClass::Euro6, "euro6"},
};

DEFINE_ENUM_IO(EcoClass, ECOCLASS_ENUM_REPRESENTATION);

std::unique_ptr<VehicleRestrictionParameters>
readVehicleRestrictionParameters(const AttrMap& attributes, TId objectId)
{
    if (std::all_of(ALL_PARAMETERS.begin(), ALL_PARAMETERS.end(),
        [&](const auto& attribute) {
            return !attributes.count(attribute);
        }))
    {
        return {};
    }
    return std::make_unique<VehicleRestrictionParameters>(
        VehicleRestrictionParameters {
            readParameter<double>(attributes, VEHICLE_RESTRICTION_WEIGHT_LIMIT_ATTR, objectId),
            readParameter<double>(attributes, VEHICLE_RESTRICTION_AXLE_WEIGHT_LIMIT_ATTR, objectId),
            readParameter<double>(attributes, VEHICLE_RESTRICTION_MAX_WEIGHT_LIMIT_ATTR, objectId),
            readParameter<double>(attributes, VEHICLE_RESTRICTION_HEIGHT_LIMIT_ATTR, objectId),
            readParameter<double>(attributes, VEHICLE_RESTRICTION_WIDTH_LIMIT_ATTR, objectId),
            readParameter<double>(attributes, VEHICLE_RESTRICTION_LENGTH_LIMIT_ATTR, objectId),
            readParameter<double>(attributes, VEHICLE_RESTRICTION_PAYLOAD_LIMIT_ATTR, objectId),
            readParameter<EcoClass>(attributes, VEHICLE_RESTRICTION_MIN_ECO_CLASS_ATTR, objectId),
            readParameter<bool>(attributes, VEHICLE_RESTRICTION_TRAILER_NOT_ALLOWED_ATTR, objectId),
        });
}

} // namespace maps::wiki::validator
