#include "helpers.h"
#include "magic_strings.h"

#include <yandex/maps/wiki/validator/common.h>
#include <yandex/maps/wiki/validator/storage/exclusion_info.h>
#include <yandex/maps/wiki/validator/storage/results_gateway.h>
#include <yandex/maps/wiki/validator/storage/message_attributes_filter.h>
#include <yandex/maps/wiki/validator/storage/stored_message_data.h>

#include <maps/wikimap/mapspro/libs/dbutils/include/parser.h>
#include <yandex/maps/wiki/revision/snapshot.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/common/include/exception.h>

#include <boost/algorithm/string/split.hpp>
#include <unordered_map>
#include <utility>

namespace maps::wiki::validator::storage {

namespace {

const char PG_ARRAY_BEGIN = '{';
const char PG_ARRAY_END = '}';
const char PG_ARRAY_DELIM = ',';

struct StoredMessageDbRow
{
    MessageId id;
    boost::optional<ExclusionInfo> exclusionInfo;
    Message message;
    bool isViewed;
};

StoredMessageData fillMessageActivity(
    const revision::Snapshot& snapshot,
    const std::vector<StoredMessageDbRow>& storedMessageRows)
{
    ObjectIdSet objectIds;
    for (const auto& row : storedMessageRows) {
        for (const RevisionID& revId : row.message.revisionIds()) {
            objectIds.insert(revId.objectId());
        }
    }

    std::unordered_map<TId, RevisionID> currentRevisionIds;

    for (const auto& revId : snapshot.objectRevisionIds(objectIds)) {
        currentRevisionIds.insert({revId.objectId(), revId});
    }

    auto isRevIdCurrent = [&](const RevisionID& revisionId)
    {
        auto it = currentRevisionIds.find(revisionId.objectId());
        if (it == currentRevisionIds.end()) {
            return revisionId.commitId() == 0;
        }
        return revisionId == it->second;
    };

    StoredMessageData result;
    for (const auto& dbRow : storedMessageRows) {
        bool active = std::all_of(
            dbRow.message.revisionIds().begin(),
            dbRow.message.revisionIds().end(),
            isRevIdCurrent);

        result.emplace_back(
            dbRow.id,
            active,
            dbRow.exclusionInfo,
            dbRow.message,
            dbRow.isViewed);
    }
    return result;
}

template<typename RowIter>
StoredMessageData
storedMessagesFromDbRows(
    const revision::Snapshot& snapshot,
    RowIter begin,
    RowIter end,
    size_t limit)
{
    std::vector<StoredMessageDbRow> storedMessageRows;

    for (auto iter = begin; iter != end; ++iter) {
        const auto& row = *iter;
        MessageId messageId(
            row[ATTRIBUTES_ID_COLUMN_NAME].template as<uint32_t>(),
            row[CONTENT_ID_COLUMN_NAME].template as<uint64_t>());

        Message message = messageFromDbRow(row);

        boost::optional<ExclusionInfo> exclusionInfo = boost::none;
        if (!row[CREATED_BY_COLUMN_NAME].is_null()) {
            exclusionInfo = ExclusionInfo{
                row[CREATED_COLUMN_NAME].template as<std::string>(),
                row[CREATED_BY_COLUMN_NAME].template as<TUId>(),
                {}
            };
        }

        bool hasViewedBy = std::any_of(
            row.begin(),
            row.end(),
            [](const pqxx::field& field) {
                return field.name() == VIEWED_BY_ARR_COLUMN_NAME;
            });

        if (hasViewedBy) {
            auto strUids = dbutils::parsePGArray(row[VIEWED_BY_ARR_COLUMN_NAME].c_str());
            for (const auto& strUid : strUids) {
                if (strUid != "NULL") {
                    exclusionInfo->viewedBy.insert(std::stoull(strUid));
                }
            }
        }
        auto it = std::find_if(row.begin(), row.end(),
            [](const auto& field) {
                return field.name() == IS_VIEWED_COLUMN_NAME;
            });
        bool isViewed = (it != row.end() && !it->is_null()) ? true : false;
        storedMessageRows.push_back(StoredMessageDbRow{
            messageId,
            std::move(exclusionInfo),
            std::move(message),
            isViewed
        });
        if (limit && storedMessageRows.size() == limit) {
            break;
        }
    }

    return fillMessageActivity(snapshot, storedMessageRows);
}

} // namespace

std::string severityToPgText(Severity severity)
{
    return std::to_string(static_cast<unsigned int>(severity));
}

Severity pgTextToSeverity(const std::string& text)
{
    unsigned long severity = std::stoul(text);
    REQUIRE(severity <= static_cast<unsigned long>(Severity::Max),
            "Unknown severity in DB: " << text);
    return static_cast<Severity>(severity);
}

std::string revisionIdsToPgText(const std::vector<RevisionID>& revIds)
{
    std::ostringstream result;
    result << PG_ARRAY_BEGIN;
    bool first = true;
    for (RevisionID revId : revIds) {
        if (!first) {
            result << PG_ARRAY_DELIM;
        } else {
            first = false;
        }

        result << revId;
    }
    result << PG_ARRAY_END;

    return result.str();
}

std::vector<RevisionID> pgTextToRevisionIds(const std::string& textStr)
{
    std::string textWithoutDelims = textStr.substr(1, textStr.length() - 2);
    if (textWithoutDelims.empty()) {
        return std::vector<RevisionID>{};
    }

    std::vector<std::string> splitted;
    boost::split(
        splitted, textWithoutDelims, [](char c) { return c == PG_ARRAY_DELIM;});
    std::vector<RevisionID> result;
    for (const std::string& revisionStr : splitted) {
        result.push_back(boost::lexical_cast<RevisionID>(revisionStr));
    }
    return result;
}

std::string whereClause(
    const MessageAttributesFilter& filter,
    const Transaction& txn)
{
    std::ostringstream result;
    result << "TRUE";

    if (filter.severity) {
        result << " AND " << SEVERITY_COLUMN_NAME
               << " = " << static_cast<int>(*filter.severity);
    }
    if (filter.checkId) {
        result << " AND " << CHECK_ID_COLUMN_NAME
               << " = " << txn.quote(*filter.checkId);
    }
    if (filter.description) {
        result << " AND " << DESCRIPTION_COLUMN_NAME
               << " = " << txn.quote(*filter.description);
    }
    if (filter.regionType) {
        result << " AND " << IMPORTANT_REGION_COLUMN_NAME << " = "
            << (*filter.regionType == RegionType::Important ? "TRUE" : "FALSE");
    }

    return result.str();
}

std::string whereClause(const geolib3::BoundingBox& bbox)
{
    std::string lowerLeft =
        "ST_MakePoint("
        + std::to_string(bbox.minX()) + "," + std::to_string(bbox.minY())
        + ")";
    std::string upperRight =
        "ST_MakePoint("
        + std::to_string(bbox.maxX()) + "," + std::to_string(bbox.maxY())
        + ")";

    return "("
        + GEOMETRY_COLUMN_NAME + " && "
        + "ST_MakeBox2D(" + lowerLeft + "," + upperRight + "))";
}

Message::Attributes messageAttributesFromDbRow(const pqxx::row& row)
{
    return Message::Attributes{
        pgTextToSeverity(row[SEVERITY_COLUMN_NAME].as<std::string>()),
        row[CHECK_ID_COLUMN_NAME].as<TCheckId>(),
        row[DESCRIPTION_COLUMN_NAME].as<std::string>(),
        row[IMPORTANT_REGION_COLUMN_NAME].as<bool>()
            ? RegionType::Important
            : RegionType::Unimportant};
}

Message messageFromDbRow(const pqxx::row& row)
{
    return Message(
        pgTextToSeverity(row[SEVERITY_COLUMN_NAME].as<std::string>()),
        row[CHECK_ID_COLUMN_NAME].as<TCheckId>(),
        row[DESCRIPTION_COLUMN_NAME].as<std::string>(),
        row[IMPORTANT_REGION_COLUMN_NAME].as<bool>()
            ? RegionType::Important
            : RegionType::Unimportant,
        row[GEOMETRY_COLUMN_NAME].is_null()
            ? std::string()
            : pqxx::binarystring(row[GEOMETRY_COLUMN_NAME]).str(),
        pgTextToRevisionIds(row[REVISION_IDS_COLUMN_NAME].as<std::string>()));
}

StoredMessageData storedMessagesFromDbRows(
    const revision::Snapshot& snapshot,
    const pqxx::result& rows)
{
    if (rows.empty()) {
        return {};
    }
    return storedMessagesFromDbRows<pqxx::result::iterator>(
        snapshot, rows.begin(), rows.end(), 0);
}

StoredMessageData storedMessagesFromDbRows(
    const revision::Snapshot& snapshot,
    const pqxx::result::iterator& begin,
    const pqxx::result::iterator& end,
    size_t limit)
{
    return storedMessagesFromDbRows<pqxx::result::iterator>(
        snapshot, begin, end, limit);
}

StoredMessageData storedMessagesFromDbRows(
    const revision::Snapshot& snapshot,
    const pqxx::result::reverse_iterator& rbegin,
    const pqxx::result::reverse_iterator& rend,
    size_t limit)
{
    return storedMessagesFromDbRows<pqxx::result::reverse_iterator>(
        snapshot, rbegin, rend, limit);
}

} // namespace maps::wiki::validator::storage
