SET search_path = service, public;

BEGIN WORK;

CREATE TABLE validation_message_stats (
    task_id integer NOT NULL,
    severity text NOT NULL,
    check_id text NOT NULL,
    description_tag text NOT NULL,
    count integer NOT NULL CHECK (count >= 0),
    PRIMARY KEY (task_id, severity, check_id, description_tag)
);


CREATE FUNCTION validation_message_stats_prune_trigger() RETURNS TRIGGER
    LANGUAGE plpgsql
    AS $$
BEGIN
    IF (NEW.count <= 0) THEN
        DELETE FROM service.validation_message_stats
            WHERE task_id = NEW.task_id
                AND severity = NEW.severity
                AND check_id = NEW.check_id
                AND description_tag = NEW.description_tag;
    END IF;
    RETURN NULL;
END;
$$;


CREATE TRIGGER prune_validation_message_stats_trigger
    AFTER UPDATE ON validation_message_stats
    FOR EACH ROW EXECUTE PROCEDURE validation_message_stats_prune_trigger();

CREATE FUNCTION validation_message_stats_update_trigger() RETURNS TRIGGER
    LANGUAGE plpgsql
    AS $$
BEGIN
    IF (TG_OP IN ('DELETE', 'UPDATE')) THEN
        UPDATE service.validation_message_stats
            SET count = count - 1
            WHERE task_id = OLD.task_id
                AND severity = OLD.severity
                AND check_id = OLD.check_id
                AND description_tag = OLD.description_tag;
    END IF;
    IF (TG_OP IN ('INSERT', 'UPDATE')) THEN
        <<upsert>>
        LOOP
            UPDATE service.validation_message_stats
                SET count = count + 1
                WHERE task_id = NEW.task_id
                    AND severity = NEW.severity
                    AND check_id = NEW.check_id
                    AND description_tag = NEW.description_tag;

            EXIT upsert WHEN found;

            BEGIN
                INSERT INTO service.validation_message_stats (
                        task_id,
                        severity,
                        check_id,
                        description_tag,
                        count)
                    VALUES (
                        NEW.task_id,
                        NEW.severity,
                        NEW.check_id,
                        NEW.description_tag,
                        1);
                EXIT upsert;

            EXCEPTION
                WHEN UNIQUE_VIOLATION THEN
                    -- retry update
            END;
        END LOOP upsert;
    END IF;

    RETURN NULL;
END;
$$;


LOCK TABLE validation_message IN EXCLUSIVE MODE;

INSERT INTO validation_message_stats
    (task_id, severity, check_id, description_tag, count)
    SELECT task_id, severity, check_id, description_tag, count(*)
        FROM validation_message
        GROUP BY task_id, severity, check_id, description_tag;

CREATE TRIGGER update_validation_message_stats_trigger
    AFTER INSERT OR UPDATE OR DELETE ON validation_message
    FOR EACH ROW EXECUTE PROCEDURE validation_message_stats_update_trigger();

COMMIT WORK;

SET search_path = public;
