CREATE SCHEMA validation;


SET search_path=validation,public;

CREATE TABLE message_attributes (
    attributes_id serial PRIMARY KEY,
    severity integer NOT NULL,
    check_id text NOT NULL,
    description text NOT NULL
);


CREATE UNIQUE INDEX message_attributes_unique_index
    ON message_attributes
    USING btree(severity, check_id, description);

CREATE OR REPLACE FUNCTION insert_message_attributes(
    p_severity integer,
    p_check_id text,
    p_description text)
RETURNS integer AS
$$
DECLARE
    r_attributes_id bigint;
BEGIN
    LOOP
        SELECT attributes_id INTO r_attributes_id
            FROM validation.message_attributes
            WHERE severity=p_severity
                  AND check_id=p_check_id
                  AND description=p_description;
        IF FOUND THEN
            RETURN r_attributes_id;
        END IF;

        BEGIN
            INSERT
                INTO validation.message_attributes
                    (severity, check_id, description)
                VALUES (p_severity, p_check_id, p_description)
                RETURNING attributes_id INTO r_attributes_id;
            RETURN r_attributes_id;
        EXCEPTION WHEN unique_violation THEN
            -- try SELECT again
        END;
    END LOOP;
END;
$$ LANGUAGE plpgsql;


CREATE TABLE message_content (
    content_id bigserial PRIMARY KEY,
    revision_ids text[] NOT NULL,
    the_geom geometry
);


CREATE OR REPLACE FUNCTION message_content_hash(
    p_revision_ids text[],
    p_the_geom geometry)
RETURNS bytea AS
$$
BEGIN
    RETURN md5(COALESCE(st_asbinary(p_the_geom), ''::bytea)
               || array_to_string(p_revision_ids, ''));
END;
$$ LANGUAGE plpgsql IMMUTABLE;

CREATE UNIQUE INDEX message_content_unique_index
    ON message_content
    USING btree(message_content_hash(revision_ids, the_geom));

CREATE INDEX message_content_the_geom_index
    ON message_content USING gist(the_geom);

CREATE OR REPLACE FUNCTION insert_message_content(
    p_revision_ids text[],
    p_the_geom geometry)
RETURNS bigint AS
$$
DECLARE
    r_content_id bigint;
BEGIN
    LOOP
        SELECT content_id INTO r_content_id
            FROM validation.message_content
            WHERE validation.message_content_hash(revision_ids, the_geom) =
                  validation.message_content_hash(p_revision_ids, p_the_geom);
        IF FOUND THEN
            RETURN r_content_id;
        END IF;

        BEGIN
            INSERT
                INTO validation.message_content(revision_ids, the_geom)
                VALUES (p_revision_ids, p_the_geom)
                RETURNING content_id INTO r_content_id;
            RETURN r_content_id;
        EXCEPTION WHEN unique_violation THEN
            -- try SELECT again
        END;
    END LOOP;
END;
$$ LANGUAGE plpgsql;


CREATE TABLE task_message (
    id bigserial PRIMARY KEY,
    task_id bigint NOT NULL,
    attributes_id integer NOT NULL,
    content_id bigint NOT NULL
);


CREATE INDEX task_message_for_filters_index
    ON task_message USING btree(attributes_id, task_id, content_id);


CREATE TABLE task_message_stats (
    task_id bigint NOT NULL,
    attributes_id integer NOT NULL,
    count integer NOT NULL CHECK (count >= 0),
    PRIMARY KEY (attributes_id, task_id)
);


CREATE FUNCTION update_task_message_stats(
    p_task_id bigint,
    p_attributes_id integer,
    p_count integer)
RETURNS integer AS
$$
DECLARE
    r_count integer;
BEGIN
    LOOP
        UPDATE validation.task_message_stats SET count = count + p_count
            WHERE task_id = p_task_id AND attributes_id = p_attributes_id
            RETURNING count INTO r_count;
        IF FOUND THEN
            RETURN r_count;
        END IF;

        BEGIN
            INSERT INTO validation.task_message_stats(task_id, attributes_id, count)
                VALUES(p_task_id, p_attributes_id, p_count);
            RETURN p_count;
        EXCEPTION
            WHEN unique_violation THEN
                -- retry update
        END;
    END LOOP;
END;
$$ LANGUAGE plpgsql;

SET search_path=public;
